<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\Commands;

use Drupal\workspaces_access\WorkspacesAccessPermissions;
use Drush\Commands\DrushCommands;
use Drush\Attributes as CLI;
use Drupal\workspaces_access\WorkspaceAccessManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\PermissionHandlerInterface;
use Drupal\workspaces_access\Hook\EntityAccess;

/**
 * Drush commands for workspaces access.
 */
class WorkspacesAccessCommands extends DrushCommands {

  /**
   * The workspace access manager.
   */
  protected WorkspaceAccessManagerInterface $workspaceAccessManager;

  /**
   * The current user.
   */
  protected AccountInterface $currentUser;

  /**
   * The workspace manager.
   */
  protected WorkspaceManagerInterface $workspaceManager;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The permission handler.
   */
  protected PermissionHandlerInterface $permissionHandler;

  /**
   * The entity access service.
   */
  protected EntityAccess $entityAccess;

  /**
   * Constructs a WorkspacesAccessCommands object.
   */
  public function __construct(
    WorkspaceAccessManagerInterface $workspaceAccessManager,
    AccountInterface $currentUser,
    WorkspaceManagerInterface $workspaceManager,
    EntityTypeManagerInterface $entityTypeManager,
    PermissionHandlerInterface $permissionHandler,
    EntityAccess $entityAccess,
  ) {
    parent::__construct();
    $this->workspaceAccessManager = $workspaceAccessManager;
    $this->currentUser = $currentUser;
    $this->workspaceManager = $workspaceManager;
    $this->entityTypeManager = $entityTypeManager;
    $this->permissionHandler = $permissionHandler;
    $this->entityAccess = $entityAccess;
  }

  /**
   * Debug permission generation.
   */
  #[CLI\Command(name: 'workspaces-access:debug-permissions')]
  #[CLI\Usage(name: 'workspaces-access:debug-permissions', description: 'Debug permission generation')]
  public function debugPermissions(): void {
    $this->io()->title('Permission Generation Debug');

    try {
      $permissions = WorkspacesAccessPermissions::permissions();

      $this->io()->text("Generated " . count($permissions) . " permissions:");

      foreach ($permissions as $key => $permission) {
        $this->io()->text("- {$key}: {$permission['title']}");
      }

      // Check if Live workspace permissions are present.
      $live_perms = array_filter($permissions, function ($key) {
        return strpos($key, 'live') !== FALSE;
      }, ARRAY_FILTER_USE_KEY);

      if (empty($live_perms)) {
        $this->io()->warning('No Live workspace permissions found!');
      }
      else {
        $this->io()->success('Live workspace permissions found:');
        foreach ($live_perms as $key => $perm) {
          $this->io()->text("  - {$key}: {$perm['title']}");
        }
      }

    }
    catch (\Exception $e) {
      $this->io()->error("Error generating permissions: {$e->getMessage()}");
      $this->io()->text("Stack trace: {$e->getTraceAsString()}");
    }
  }

  /**
   * Check permissions for current user in current workspace.
   */
  #[CLI\Command(name: 'workspaces-access:check-permissions')]
  #[CLI\Usage(name: 'workspaces-access:check-permissions', description: 'Check workspace permissions for current user')]
  public function checkPermissions(): void {
    $account = $this->currentUser;
    $workspace_manager = $this->workspaceManager;

    if (!$workspace_manager->hasActiveWorkspace()) {
      $this->io()->error('No active workspace');
      return;
    }

    $workspace = $workspace_manager->getActiveWorkspace();
    $this->io()->title('Workspace Access Check');
    $this->io()->text("Current workspace: {$workspace->label()} ({$workspace->id()})");
    $this->io()->text("Current user: {$account->getDisplayName()} ({$account->id()})");

    $permissions = [
      'workspace_live_view_content' => '@ Live - View content',
      'workspace_live_add_content' => '@ Live - Add content',
      'workspace_live_edit_content' => '@ Live - Edit content',
      'workspace_live_remove_content' => '@ Live - Remove content',
      "workspace_{$workspace->id()}_view_content" => "{$workspace->label()} - View content",
      "workspace_{$workspace->id()}_add_content" => "{$workspace->label()} - Add content",
      "workspace_{$workspace->id()}_edit_content" => "{$workspace->label()} - Edit content",
      "workspace_{$workspace->id()}_remove_content" => "{$workspace->label()} - Remove content",
    ];

    // Show all available workspaces.
    $this->io()->section('Available Workspaces');
    $workspace_storage = $this->entityTypeManager->getStorage('workspace');
    $all_workspaces = $workspace_storage->loadMultiple();
    foreach ($all_workspaces as $ws) {
      $this->io()->text("- {$ws->label()} (ID: {$ws->id()})");
    }

    $this->io()->section('Permissions Check');
    foreach ($permissions as $perm => $label) {
      $has_perm = $account->hasPermission($perm);
      $status = $has_perm ? '<fg=green>YES</>' : '<fg=red>NO</>';
      $this->io()->text("{$label}: {$status}");

      // Debug: Check if permission exists in system.
      $all_system_perms = $this->permissionHandler->getPermissions();
      $perm_exists = isset($all_system_perms[$perm]);
      $exists_status = $perm_exists ? '<fg=green>EXISTS</>' : '<fg=red>MISSING</>';
      $this->io()->text("  Permission exists in system: {$exists_status}");
    }

    // Test access on a sample node if possible.
    try {
      $node_storage = $this->entityTypeManager->getStorage('node');
      $nodes = $node_storage->loadByProperties([], ['limit' => 1]);
      if (!empty($nodes)) {
        $node = reset($nodes);
        $this->io()->section('Access Test on Sample Node');
        $this->io()->text("Testing node: {$node->getTitle()} ({$node->id()})");

        $operations = ['view', 'update', 'delete'];
        foreach ($operations as $op) {
          $access = $node->access($op, $account);
          $status = $access ? '<fg=green>ALLOWED</>' : '<fg=red>DENIED</>';
          $this->io()->text("{$op}: {$status}");

          // Also test our direct hook.
          try {
            $direct_access = $this->entityAccess->entityAccess($node, $op, $account);
            $direct_status = $direct_access->isAllowed() ? 'ALLOWED' : ($direct_access->isForbidden() ? 'FORBIDDEN' : 'NEUTRAL');
            $this->io()->text("  Direct hook result: {$direct_status}");
          }
          catch (\Exception $e) {
            $this->io()->text("  Direct hook error: {$e->getMessage()}");
          }
        }
      }
    }
    catch (\Exception $e) {
      $this->io()->warning("Could not test node access: {$e->getMessage()}");
    }

    // Show all user permissions for debugging.
    $this->io()->section('All User Permissions');
    $all_perms = $account->getRoles();
    $this->io()->text("User roles: " . implode(', ', $all_perms));

    // Get role permissions.
    $role_storage = $this->entityTypeManager->getStorage('user_role');
    foreach ($all_perms as $role_id) {
      if ($role_id === 'anonymous') {
        continue;
      }
      $role = $role_storage->load($role_id);
      if ($role) {
        $role_perms = $role->getPermissions();
        $workspace_perms = array_filter($role_perms, function ($perm) {
          return strpos($perm, 'workspace') !== FALSE || strpos($perm, 'disable') !== FALSE;
        });
        if (!empty($workspace_perms)) {
          $this->io()->text("Role '{$role_id}' workspace permissions: " . implode(', ', $workspace_perms));
        }
      }
    }
  }

}
