<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\Controller;

use Drupal\Core\Url;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\PermissionHandlerInterface;

/**
 * Controller for workspaces access debugging.
 */
class WorkspacesAccessController extends ControllerBase {

  use StringTranslationTrait;

  /**
   * The permission handler.
   */
  protected PermissionHandlerInterface $permissionHandler;

  /**
   * Constructs a WorkspacesAccessController object.
   */
  public function __construct(PermissionHandlerInterface $permissionHandler) {
    $this->permissionHandler = $permissionHandler;
  }

  /**
   * Debug page to show workspace permissions.
   */
  public function debug() {
    $build = [];

    // Get workspace count for testing.
    $workspace_storage = $this->entityTypeManager()->getStorage('workspace');
    $workspaces = $workspace_storage->loadMultiple();
    $test_result = count($workspaces);

    $build['module_test'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<h3>Module Test Result: @result workspaces found</h3>', [
        '@result' => $test_result,
      ]),
    ];

    // Get all workspaces.
    $workspace_storage = $this->entityTypeManager()->getStorage('workspace');
    $workspaces = $workspace_storage->loadMultiple();

    $build['workspaces'] = [
      '#type' => 'table',
      '#caption' => $this->t('Workspaces Found (@count)', ['@count' => count($workspaces)]),
      '#header' => [
        $this->t('Workspace ID'),
        $this->t('Label'),
        $this->t('Permission Key'),
      ],
      '#rows' => [],
    ];

    foreach ($workspaces as $workspace) {
      $workspace_id = $workspace->id();
      $workspace_label = $workspace->label();
      $permission_key = "workspace_{$workspace_id}_edit_content (and add/remove variants)";

      $build['workspaces']['#rows'][] = [
        $workspace_id,
        $workspace_label,
        $permission_key,
      ];
    }

    // Get current permissions.
    $permission_handler = $this->getPermissionHandler();
    $all_permissions = $permission_handler->getPermissions();

    $workspace_permissions = array_filter($all_permissions, function ($key) {
      return strpos($key, 'workspace_') === 0;
    }, ARRAY_FILTER_USE_KEY);

    $build['permissions'] = [
      '#type' => 'table',
      '#caption' => $this->t('Generated Permissions (@count)', ['@count' => count($workspace_permissions)]),
      '#header' => [
        $this->t('Permission Key'),
        $this->t('Title'),
        $this->t('Description'),
      ],
      '#rows' => [],
    ];

    foreach ($workspace_permissions as $key => $permission) {
      $build['permissions']['#rows'][] = [
        $key,
        $permission['title'] ?? '',
        $permission['description'] ?? '',
      ];
    }

    // Add manual permission generation test.
    $build['manual_test'] = [
      '#type' => 'details',
      '#title' => $this->t('Manual Permission Generation Test'),
      '#open' => TRUE,
    ];

    // Manually call hook_permissions to test.
    $manual_permissions = workspaces_access_permissions();

    $build['manual_test']['content'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p>Manual hook_permissions() call returned @count permissions:</p>', [
        '@count' => count($manual_permissions),
      ]),
    ];

    if (!empty($manual_permissions)) {
      $permission_list = '<ul>';
      foreach ($manual_permissions as $key => $perm) {
        $permission_list .= '<li><strong>' . $key . ':</strong> ' . ($perm['title'] ?? 'No title') . '</li>';
      }
      $permission_list .= '</ul>';
      $build['manual_test']['content']['#markup'] .= $permission_list;
    }

    // Add force rediscovery button.
    $build['force_rediscovery'] = [
      '#type' => 'details',
      '#title' => $this->t('Force Hook Rediscovery'),
      '#open' => TRUE,
    ];

    $build['force_rediscovery']['content'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p>If permissions are not showing up, click this button to force Drupal to rediscover hooks:</p>'),
    ];

    $build['force_rediscovery']['button'] = [
      '#type' => 'link',
      '#title' => $this->t('Force Rediscovery'),
      '#url' => Url::fromRoute('workspaces_access.force_rediscovery'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    return $build;
  }

  /**
   * Force hook rediscovery and redirect back to debug page.
   */
  public function forceRediscovery() {
    workspaces_access_force_rediscovery();

    $this->messenger()->addMessage($this->t('Hook rediscovery forced. Check the debug page and permissions.'));

    return $this->redirect('workspaces_access.debug');
  }

  /**
   * Gets the permission handler service.
   */
  protected function getPermissionHandler() {
    return $this->permissionHandler;
  }

}
