<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\EventSubscriber;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\workspaces\Event\WorkspacePostPublishEvent;
use Drupal\workspaces\Event\WorkspacePrePublishEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for workspace access management.
 */
class WorkspaceSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected CacheTagsInvalidatorInterface $cacheTagsInvalidator,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      WorkspacePostPublishEvent::class => 'onWorkspacePostPublish',
      WorkspacePrePublishEvent::class => 'onWorkspacePrePublish',
    ];
  }

  /**
   * Reacts to workspace post-publish events.
   *
   * @param \Drupal\workspaces\Event\WorkspacePostPublishEvent $event
   *   The workspace post-publish event.
   */
  public function onWorkspacePostPublish(WorkspacePostPublishEvent $event): void {
    $this->invalidatePermissionCache();
  }

  /**
   * Reacts to workspace pre-publish events.
   *
   * @param \Drupal\workspaces\Event\WorkspacePrePublishEvent $event
   *   The workspace pre-publish event.
   */
  public function onWorkspacePrePublish(WorkspacePrePublishEvent $event): void {
    $this->invalidatePermissionCache();
  }

  /**
   * Invalidates permission-related caches.
   */
  protected function invalidatePermissionCache(): void {
    // Invalidate permission caches to ensure new permissions are recognized.
    $this->cacheTagsInvalidator->invalidateTags(['config:workspaces_access.permissions']);
  }

}
