<?php

declare(strict_types=1);

namespace Drupal\workspaces_access;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\workspaces\WorkspaceRepositoryInterface;

/**
 * Manages workspace access permissions.
 */
class WorkspaceAccessManager implements WorkspaceAccessManagerInterface {

  use StringTranslationTrait;

  public function __construct(
    protected WorkspaceRepositoryInterface $workspaceRepository,
  ) {}

  /**
   * Gets all workspace permissions.
   *
   * @return array
   *   An array of permission definitions.
   */
  public function getWorkspacePermissions(): array {
    $permissions = [];

    // Get all workspaces.
    $workspaces = $this->workspaceRepository->loadMultiple();

    foreach ($workspaces as $workspace) {
      $workspace_id = $workspace->id();
      $workspace_label = $workspace->label();

      // Determine ordering prefix based on workspace type.
      $is_uuid = preg_match('/^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/', $workspace_id);
      $order_prefix = $is_uuid ? '3' : '2';

      // Add permissions for each operation.
      $operations = [
        'add' => $this->t(
           '@prefix - Workspace @workspace - Add content',
           [
             '@prefix' => $order_prefix,
             '@workspace' => $workspace_label,
           ]
        ),
        'edit' => $this->t(
           '@prefix - Workspace @workspace - Edit content',
           [
             '@prefix' => $order_prefix,
             '@workspace' => $workspace_label,
           ]
        ),
        'remove' => $this->t(
           '@prefix - Workspace @workspace - Remove content',
           [
             '@prefix' => $order_prefix,
             '@workspace' => $workspace_label,
           ]
        ),
      ];

      foreach ($operations as $op => $title) {
        $permissions["workspace_{$workspace_id}_{$op}_content"] = [
          'title' => $title,
          'description' => $this->t('Allow @op content within the @workspace workspace.', [
            '@op' => $op . 'ing',
            '@workspace' => $workspace_label,
          ]),
          'restrict access' => TRUE,
        ];
      }
    }

    return $permissions;
  }

  /**
   * Gets permission for a specific workspace and operation.
   *
   * @param string $workspace_id
   *   The workspace ID.
   * @param string $operation
   *   The operation (add, edit, remove).
   *
   * @return string
   *   The permission string.
   */
  public function getWorkspacePermission(string $workspace_id, string $operation = 'edit'): string {
    return "workspace_{$workspace_id}_{$operation}_content";
  }

}
