<?php

declare(strict_types=1);

namespace Drupal\Tests\workspaces_access\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\workspaces\Entity\Workspace;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Tests access control for workspaces access module.
 *
 * @group workspaces_access
 */
class WorkspacesAccessTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'system',
    'workspaces',
    'workspaces_access',
    'node',
    'field',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('workspaces', ['workspace_association']);
    $this->installSchema('node', ['node_access']);
    $this->installSchema('user', ['users_data']);

    $this->installEntitySchema('workspace');
    $this->installEntitySchema('user');
    $this->installEntitySchema('node');

    // Create a basic node type for testing.
    $node_type = NodeType::create(
      [
        'type' => 'article',
        'name' => 'Article',
      ]
    );
    $node_type->save();

    // Create workspaces.
    Workspace::create(['id' => 'live', 'label' => 'Live'])->save();
    Workspace::create(['id' => 'staging', 'label' => 'Staging'])->save();
    Workspace::create(['id' => 'development', 'label' => 'Development'])->save();
  }

  /**
   * Tests Live workspace access control without Live permissions.
   */
  public function testLiveWorkspaceWithoutPermissions(): void {
    // Create a user with basic content permissions but no Live workspace
    // permissions.
    $user = $this->createUser(
      [
        'create article content',
        'edit any article content',
        'delete any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Live workspace.
    $workspace = Workspace::load('live');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // User should NOT be able to create in Live workspace without create
    // permission.
    $entityTypeManager = \Drupal::entityTypeManager();
    $createAccess = $entityTypeManager->getAccessControlHandler('node')
      ->createAccess('article', $user, [], TRUE);
    $this->assertFalse($createAccess->isAllowed(),
      'User without Live create permission should not be able to create in Live workspace');

    // User should NOT be able to edit in Live workspace without edit
    // permission.
    $this->assertFalse($node->access('update', $user),
      'User without Live edit permission should not be able to edit in Live workspace');

    // User should NOT be able to delete in Live workspace without delete
    // permission.
    $this->assertFalse($node->access('delete', $user),
      'User without Live delete permission should not be able to delete in Live workspace');
  }

  /**
   * Tests Live workspace create access control with Live create permission.
   */
  public function testLiveWorkspaceCreateWithPermission(): void {
    // Create a user with Live workspace create permission, view permission,
    // and content permissions.
    $user = $this->createUser(
      [
        'workspace_live_add_content',
        'workspace_live_view_content',
        'create article content',
      ]
    );
    // Note: Permission title is 'Live - Add content' but machine name remains
    // the same.
    $this->setCurrentUser($user);

    // Switch to Live workspace.
    $workspace = Workspace::load('live');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // Test entity creation access.
    $entityTypeManager = \Drupal::entityTypeManager();
    $accessResult = $entityTypeManager->getAccessControlHandler('node')
      ->createAccess('article', $user, [], TRUE);
    $this->assertTrue($accessResult->isAllowed(),
      'User with Live create permission should be able to create in Live workspace');
  }

  /**
   * Tests Live workspace edit access control with Live edit permission.
   */
  public function testLiveWorkspaceEditWithPermission(): void {
    // Create a user with Live workspace edit permission and content
    // permissions.
    $user = $this->createUser(
      [
        'workspace_live_edit_content',
        'edit any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Live workspace.
    $workspace = Workspace::load('live');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // User SHOULD be able to edit in Live workspace with the Live edit
    // permission.
    $this->assertTrue($node->access('update', $user),
      'User with Live edit permission should be able to edit in Live workspace');
  }

  /**
   * Tests Live workspace delete access control with Live delete permission.
   */
  public function testLiveWorkspaceDeleteWithPermission(): void {
    // Create a user with Live workspace delete permission and content
    // permissions.
    $user = $this->createUser(
      [
        'workspace_live_remove_content',
        'delete any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Live workspace.
    $workspace = Workspace::load('live');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // User SHOULD be able to delete in Live workspace with the Live delete
    // permission.
    $this->assertTrue($node->access('delete', $user), 'User with Live delete permission should be able to delete in Live workspace');
  }

  /**
   * Tests that Live workspace permissions require content permissions.
   */
  public function testLiveWorkspaceRequiresContentPermissions(): void {
    // Create users with Live workspace permissions but no content permissions.
    $createUser = $this->createUser(
      [
        'workspace_live_add_content',
        'workspace_live_view_content',
        // No content permissions granted.
      ]
    );
    $editUser = $this->createUser(
      [
        'workspace_live_edit_content',
        'workspace_live_view_content',
        // No content permissions granted.
      ]
    );
    $deleteUser = $this->createUser(
      [
        'workspace_live_remove_content',
        // No content permissions granted.
      ]
    );

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Live workspace.
    $workspace = Workspace::load('live');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // Test create permission without content permissions.
    $this->setCurrentUser($createUser);
    $entityTypeManager = \Drupal::entityTypeManager();
    $createAccess = $entityTypeManager->getAccessControlHandler('node')
      ->createAccess('article', $createUser, [], TRUE);
    $this->assertFalse($createAccess->isAllowed(),
      'User with Live create permission but no content permissions should not be able to create');

    // Test edit permission without content permissions.
    $this->setCurrentUser($editUser);
    $this->assertFalse($node->access('update', $editUser),
      'User with Live edit permission but no content permissions should not be able to edit');

    // Test delete permission without content permissions.
    $this->setCurrentUser($deleteUser);
    $this->assertFalse($node->access('delete', $deleteUser),
      'User with Live delete permission but no content permissions should not be able to delete');
  }

  /**
   * Tests Live workspace access requires explicit permission.
   */
  public function testLiveWorkspaceRequiresExplicitPermission(): void {
    // Create a user with basic content permissions but no Live workspace
    // permission.
    $user = $this->createUser(
      [
        'create article content',
        'edit any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Live workspace.
    $workspace = Workspace::load('live');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // User should NOT be able to edit in Live workspace without explicit
    // permission.
    $this->assertFalse($node->access('update', $user), 'User without Live workspace permission should not be able to edit in Live workspace');
  }

  /**
   * Tests users with staging create permission can create in staging workspace.
   */
  public function testStagingWorkspaceCreateAccessControl(): void {
    // Create a user with staging create permission and live view permission
    // (required for creating content in workspace as draft is created in live).
    $user = $this->createUser(
      [
        'workspace_staging_add_content',
        'workspace_staging_view_content',
        'workspace_live_view_content',
        'create article content',
      ]
    );
    $this->setCurrentUser($user);

    // Switch to Staging workspace.
    $staging_workspace = Workspace::load('staging');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($staging_workspace);

    // Test entity creation access.
    $entityTypeManager = \Drupal::entityTypeManager();
    $accessResult = $entityTypeManager->getAccessControlHandler('node')->createAccess('article', $user, [], TRUE);
    $this->assertTrue($accessResult->isAllowed(),
      'User with Staging create permission should be able to create in Staging workspace');
  }

  /**
   * Tests users with staging edit permission can edit in staging workspace.
   */
  public function testStagingWorkspaceEditAccessControl(): void {
    // Create a user with staging edit permission.
    $user = $this->createUser(
      [
        'workspace_staging_edit_content',
        'workspace_staging_view_content',
        'edit any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Staging workspace.
    $staging_workspace = Workspace::load('staging');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($staging_workspace);

    // User SHOULD be able to edit in Staging workspace.
    $this->assertTrue($node->access('update', $user),
      'User with Staging edit permission should be able to edit in Staging workspace');
  }

  /**
   * Tests users with staging delete permission can delete in staging workspace.
   */
  public function testStagingWorkspaceDeleteAccessControl(): void {
    // Create a user with staging delete permission.
    $user = $this->createUser(
      [
        'workspace_staging_remove_content',
        'delete any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Staging workspace.
    $staging_workspace = Workspace::load('staging');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($staging_workspace);

    // User SHOULD be able to delete in Staging workspace.
    $this->assertTrue($node->access('delete', $user), 'User with Staging delete permission should be able to delete in Staging workspace');
  }

  /**
   * Tests users with only create permission cannot edit or delete in staging.
   */
  public function testStagingWorkspacePartialPermissions(): void {
    // Create a user with only create permission and live view permission
    // (required for creating content in workspace).
    $user = $this->createUser(
      [
        'workspace_staging_add_content',
        'workspace_staging_view_content',
        'workspace_live_view_content',
        'create article content',
        'edit any article content',
        'delete any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Switch to Staging workspace.
    $staging_workspace = Workspace::load('staging');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($staging_workspace);

    // User can create but cannot edit or delete.
    $entityTypeManager = \Drupal::entityTypeManager();
    $createAccess = $entityTypeManager->getAccessControlHandler('node')
      ->createAccess('article', $user, [], TRUE);
    $this->assertTrue($createAccess->isAllowed(), 'User with create permission should be able to create');

    $this->assertFalse($node->access('update', $user), 'User with only create permission should not be able to edit');
    $this->assertFalse($node->access('delete', $user), 'User with only create permission should not be able to delete');
  }

  /**
   * Tests users without any workspace permissions cannot edit in any workspace.
   */
  public function testNoWorkspacePermissionsBlocksAllAccess(): void {
    // Create a user with content permissions but no workspace permissions.
    $user = $this->createUser(
      [
        'create article content',
        'edit any article content',
      ]
    );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
      [
        'type' => 'article',
        'title' => 'Test Article',
      ]
    );
    $node->save();

    // Test in Live workspace.
    $live_workspace = Workspace::load('live');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($live_workspace);
    $this->assertFalse($node->access('update', $user), 'User without workspace permissions should not be able to edit in Live workspace');

    // Test in Staging workspace.
    $staging_workspace = Workspace::load('staging');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($staging_workspace);
    $this->assertFalse($node->access('update', $user), 'User without workspace permissions should not be able to edit in Staging workspace');
  }

}
