# Workspaces Access

A Drupal module that provides controlled access to content operations in the Live workspace.

## Overview

The Workspaces Access module provides essential security controls for the Live workspace in Drupal's Workspaces system. It ensures that users must have explicit permissions to perform content operations (view, create, edit, delete) in the production Live workspace, preventing unauthorized modifications to production content.

## Features

- **Live Workspace Protection**: Explicit permissions required for all content operations in the Live workspace
- **Operation-Specific Permissions**: Separate permissions for view, add, edit, and remove operations in Live workspace
- **Security-First Approach**: Prevents unauthorized access to production content
- **Simple Permission Model**: Focused on protecting the critical Live workspace environment

## Installation

1. Place the `workspaces_access` module in your `web/modules/contrib/` directory
2. Enable the module: `drush en workspaces_access`
3. Clear caches: `drush cr`

## Testing

This module includes comprehensive test coverage. To run the tests:

### Prerequisites

Ensure your Drupal project has testing dependencies installed (phpunit, drupal/core-dev, etc.).

### Running Tests

Run all tests:
```bash
# From the project root directory
phpunit web/modules/contrib/workspaces_access

# Run specific test classes
phpunit web/modules/contrib/workspaces_access/tests/src/Functional/WorkspacesAccessTest.php
phpunit web/modules/contrib/workspaces_access/tests/src/Functional/WorkspaceAccessEventTest.php
```

### Test Structure

- **Functional Tests** (`tests/src/Functional/`): Full Drupal environment tests covering access control, permissions, and workspace operations

### Test Dependencies

This module is designed to work within a larger Drupal project. Testing dependencies should be managed at the project level:

- `phpunit/phpunit`: Unit testing framework
- `drupal/core-dev`: Drupal core development tools
- `drupal/test_tools`: Drupal testing utilities

Ensure these are installed in your main project before running tests.

## Usage

### Permission Assignment

After installation, Live workspace permissions will appear in Drupal's permission administration interface. You can access them at:

- **Direct Link**: Navigate to `/admin/people/permissions`

#### Available Permissions

**Important:** To perform content operations in the Live workspace, users need BOTH the Live workspace permission AND the corresponding content permission. For example:
- Add: "Live - Add content" + "create article content"
- Edit: "Live - Edit content" + "edit any article content"
- View: "Live - View content" + "view article content"
- Delete: "Live - Remove content" + "delete any article content"

**Live Workspace Permissions:**
- "@ Live - View content" - Role can view content in Live workspace. Requires relevant permissions (e.g., "view article content") for operations.
- "@ Live - Add content" - Role can create content in Live workspace. Requires relevant permissions (e.g., "create article content") for operations.
- "@ Live - Edit content" - Role can edit content in Live workspace. Requires relevant permissions (e.g., "edit article content") for operations.
- "@ Live - Remove content" - Role can delete content in Live workspace. Requires relevant permissions (e.g., "delete article content") for operations.

**Note:** Non-Live workspaces do not have permission restrictions. Users with appropriate content permissions can perform operations in any non-Live workspace.

### How It Works

#### Live Workspace (Protected)
1. **Permission Check**: Users must have the specific Live workspace permission for the operation they want to perform
2. **Content Permission Check**: Users must also have the corresponding Drupal content permission
3. **Access Control**: If either permission is missing, access is explicitly denied
4. **Security**: This prevents unauthorized modifications to production content

#### Non-Live Workspaces (Unrestricted)
- No additional permission checks are applied
- Standard Drupal content permissions control access
- Users with appropriate content permissions can work freely in development/staging workspaces

### Example Scenarios

#### Scenario 1: Standard User
- User has basic content permissions (create, edit, delete)
- User does NOT have any Live workspace permissions

Result: **Explicitly denied** access to create, edit, or delete in Live workspace

#### Scenario 2: Content Editor
- User has basic content permissions
- User has "Live - View content" and "Live - Edit content" permissions

Result: **Can** view and edit existing content in Live workspace

#### Scenario 3: Administrator
- User has all Live workspace permissions ("Live - View/Add/Edit/Remove content")
- User has basic content permissions

Result: **Can** perform all operations in Live workspace

#### Scenario 4: Developer
- User has basic content permissions
- User does NOT have Live workspace permissions

Result:
- **Can** work freely in non-Live workspaces (staging, development, etc.)
- **Cannot** modify content in Live workspace

#### Scenario 5: Content Creator
- User has "Live - View content" and "Live - Add content" permissions
- User has basic content creation permissions

Result:
- Can view existing content and create new content in Live workspace
- Cannot edit or delete existing content in Live workspace

## Field-Based Permissions

The Workspaces Access module uses field-based permission storage for enhanced flexibility and persistence across workspace operations.

### Permission Fields

Each workspace entity includes the following base fields for role-based access control:

- `field_workspace_roles_view`: Roles that can view content in this workspace
- `field_workspace_roles_add`: Roles that can create content in this workspace
- `field_workspace_roles_edit`: Roles that can edit content in this workspace
- `field_workspace_roles_remove`: Roles that can delete content in this workspace

### Permission Assignment

Permissions are assigned at the workspace level by adding user roles to these fields. This provides:

- **Granular Control**: Different roles can have different permissions per workspace
- **Persistence**: Permissions are maintained when workspaces are cloned or published
- **Flexibility**: Easy to modify permissions without code changes

### Permission Priority

1. **Workspace Role Permissions**: Checked first for non-Live workspaces
2. **Live Workspace Permissions**: Global permissions for the Live workspace
3. **Content Permissions**: Standard Drupal content permissions (create, edit, delete)
4. **Event System**: Custom business logic can override any of the above

## Debug Interface

The module provides a comprehensive debug interface for troubleshooting permission issues.

### Accessing the Debug Page

Navigate to `/admin/config/workflow/workspaces/debug` or use the "Workspaces access" tab in the Workspaces administration section.

### Debug Features

- **Workspace Role Assignments**: View all role assignments for each workspace
- **Field Information**: Details about permission fields and their storage
- **Permission Testing**: Test permission generation and validation
- **Force Rediscovery**: Button to refresh Drupal's hook implementations

### Drush Commands

The module includes Drush commands for command-line management:

#### Debug Permissions
```bash
drush workspaces-access:debug-permissions
```
Shows generated permissions and validates Live workspace permissions.

#### Check User Permissions
```bash
drush workspaces-access:check-permissions
```
Displays workspace permissions for the current user in the current workspace context.

## Workspace Cloning Integration

The module integrates with the WSE (Workspace Entity) module for seamless permission handling during workspace operations.

### Publishing Behavior

When a workspace is published:

1. The workspace status changes to 'closed'
2. Content is merged to the Live workspace
3. The original workspace is deleted
4. A new workspace is created with the same name but different machine ID
5. Permission metadata is automatically copied to the new workspace

### Cloning Support

- **Automatic Detection**: Identifies newly cloned workspaces by creation time and naming
- **Permission Preservation**: Copies all role assignments to cloned workspaces
- **Logging**: Comprehensive logging of permission copy operations
- **Configuration**: Respects WSE cloning settings

## Custom Business Logic Integration

The Workspaces Access module provides an extensible event system that allows you to implement custom business-specific access restrictions in your own modules.

### Adding Business-Specific Restrictions

You can create custom modules that subscribe to workspace access events to implement your own business rules. This allows for clean separation between generic workspace permissions and project-specific access control logic.

#### 1. Create an Event Subscriber

Create a new class that implements `EventSubscriberInterface`:

```php
<?php

declare(strict_types=1);

namespace Drupal\my_business_module\EventSubscriber;

use Drupal\workspaces_access\Event\WorkspaceAccessEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for custom workspace access restrictions.
 */
class WorkspaceAccessSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'workspaces_access.workspace_check' => 'onWorkspaceAccessCheck',
    ];
  }

  /**
   * Implements custom workspace access restrictions.
   *
   * @param \Drupal\workspaces_access\Event\WorkspaceAccessEvent $event
   *   The workspace access event.
   */
  public function onWorkspaceAccessCheck(WorkspaceAccessEvent $event): void {
    $workspaceName = $event->getWorkspaceName();
    $operation = $event->getOperation();
    $entityTypeId = $event->getEntityTypeId();
    $bundle = $event->getBundle();

    // Only apply restrictions to editing operations
    if (!in_array($operation, ['create', 'update', 'delete'])) {
      return;
    }

    // Example: Block certain content types in specific workspaces
    if ($workspaceName === 'production' && $bundle === 'unpublished_content') {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::forbidden(
          'Cannot edit unpublished content in production workspace'
        )->cachePerPermissions()
      );
      return;
    }

    // Example: Allow specific operations in certain workspaces
    if ($workspaceName === 'staging' && $entityTypeId === 'media' && $operation === 'create') {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::allowed(
          'Media creation allowed in staging workspace'
        )->cachePerPermissions()
      );
      return;
    }
  }

}
```

#### 2. Register Your Event Subscriber

Create a service definition in your module's `services.yml` file:

```yaml
# my_business_module.services.yml
services:
  my_business_module.workspace_access_subscriber:
    class: Drupal\my_business_module\EventSubscriber\WorkspaceAccessSubscriber
    tags:
      - { name: event_subscriber }
```

#### 3. Enable Your Module

Enable your custom module and the event subscriber will automatically be registered.

### Real-World Example: RAGS/CAT Restrictions

Here's an example of how the PDC module implements business-specific restrictions:

```php
<?php

declare(strict_types=1);

namespace Drupal\pdc\EventSubscriber;

use Drupal\workspaces_access\Event\WorkspaceAccessEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for PDC-specific workspace access restrictions.
 */
class WorkspaceAccessSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'workspaces_access.workspace_check' => 'onWorkspaceAccessCheck',
    ];
  }

  /**
   * Implements PDC-specific workspace access restrictions.
   */
  public function onWorkspaceAccessCheck(WorkspaceAccessEvent $event): void {
    $workspaceName = $event->getWorkspaceName();
    $operation = $event->getOperation();
    $entityTypeId = $event->getEntityTypeId();
    $bundle = $event->getBundle();

    // Only apply restrictions to editing operations
    if (!in_array($operation, ['create', 'update', 'delete'])) {
      return;
    }

    $typeToCheck = !empty($bundle) ? $bundle : $entityTypeId;

    // Business rule: RAGS workspaces cannot access CAT content
    if (stripos($workspaceName, 'rags') !== false &&
        stripos($typeToCheck, 'cat') !== false) {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::forbidden(
          'Access blocked: workspace contains "rags" and content contains "cat"'
        )->cachePerPermissions()
      );
      return;
    }

    // Business rule: CAT workspaces cannot access RAGS content
    if (stripos($workspaceName, 'cat') !== false &&
        stripos($typeToCheck, 'rags') !== false) {
      $event->setAccessResult(
        \Drupal\Core\Access\AccessResult::forbidden(
          'Access blocked: workspace contains "cat" and content contains "rags"'
        )->cachePerPermissions()
      );
      return;
    }
  }

}
```

### Event System Benefits

- **Separation of Concerns**: Keep generic workspace logic separate from business rules
- **Extensibility**: Multiple modules can subscribe to the same event
- **Override Capability**: Business rules can override or extend base permissions
- **Clean Architecture**: Event-driven design for maintainable code
- **Performance**: Events are only dispatched when needed

### Available Event Data

The `WorkspaceAccessEvent` provides access to:

- `$event->getWorkspaceName()`: Current workspace name
- `$event->getOperation()`: Operation being performed (view, create, update, delete)
- `$event->getEntityTypeId()`: Entity type machine name
- `$event->getBundle()`: Entity bundle (content type, media type, etc.)
- `$event->getEntity()`: The actual entity object (when available)
- `$event->getAccount()`: The user account performing the operation
- `$event->setAccessResult(AccessResultInterface)`: Set the access result

## Technical Details

### Permission Logic

The module implements multiple hooks for comprehensive access control:

1. **`hook_entity_access()`**: Controls access to existing entities
2. **`hook_entity_create_access()`**: Controls creation of new entities
3. **`hook_node_access()`**: Node-specific access control (respects node access modules)

### Permission Storage

The module uses a hybrid approach to permission storage:

- **Live Workspace**: Static permissions in Drupal's permission system
- **Non-Live Workspaces**: Field-based storage on workspace entities
- **Fallback**: Graceful degradation when fields are not available

### Field-Based Storage Benefits

- **Persistence**: Permissions survive workspace recreation during publishing
- **Performance**: No need to regenerate permissions on every request
- **Flexibility**: Easy to modify permissions programmatically
- **Integration**: Works seamlessly with WSE workspace cloning

### Access Control Flow

1. **Entity Support Check**: Verify entity type is supported by workspaces
2. **Workspace Context**: Determine current workspace (Live or non-Live)
3. **Event Dispatch**: Allow custom business logic to override permissions
4. **Permission Check**: Check appropriate permissions based on workspace type
5. **Content Permission**: Verify standard Drupal content permissions
6. **Access Decision**: Return allowed, forbidden, or neutral result

### Caching Strategy

- **Permission Caching**: Uses Drupal's permission caching system
- **Context-Aware**: Cache contexts ensure proper invalidation
- **Tag-Based**: Cache tags for efficient invalidation on workspace changes
- **Performance**: Minimizes database queries through intelligent caching

## Dependencies

### Required
- **Drupal Core**: ^10 || ^11
- **Workspaces module**: Core module for workspace functionality

### Optional
- **WSE (Workspace Entity)**: Enhanced workspace cloning and lifecycle management
- **Custom modules**: Can subscribe to workspace access events for business logic

## Compatibility

This module is designed to work alongside existing workspace functionality and respects:
- Workspace ownership and lifecycle
- Existing access patterns and permissions
- Core workspace permissions and behaviors
- WSE module integration for advanced workspace operations
- Custom event subscribers for business-specific logic

## Testing

The module includes comprehensive tests covering:
- Permission generation
- Access control logic
- Workspace lifecycle events
- Edge cases and error conditions

## Security Considerations

- **Live Workspace Protection**: Explicit permissions required for all operations in production
- **Controlled Access**: Administrators can grant Live workspace access to specific roles as needed
- **Principle of Least Privilege**: Live workspace requires explicit permissions, preventing accidental production edits
- **Restrictive Permissions**: All Live workspace permissions are marked as "restrict access" for enhanced security
- **Cache Security**: Proper cache contexts prevent permission leakage
- **Entity-Level Enforcement**: Access control is enforced at the entity level for comprehensive protection
- **Workspace Freedom**: Non-Live workspaces remain unrestricted for development workflow efficiency

## Support

For issues and feature requests, please create an issue in the project repository.
