<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\Hook;

use Drupal\node\NodeInterface;
use Drupal\workspaces_access\Event\WorkspaceAccessEvent;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\workspaces\WorkspaceInformationInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Defines a class for reacting to entity access control hooks.
 */
class EntityAccess {

  public function __construct(
    protected WorkspaceManagerInterface $workspaceManager,
    protected WorkspaceInformationInterface $workspaceInfo,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EventDispatcherInterface $eventDispatcher,
  ) {
  }

  /**
   * Check access blocked based on Live workspace restrictions.
   *
   * This method dispatches events to allow other modules to implement custom
   * Live workspace-specific access restrictions.
   *
   * @param string $workspaceId
   *   The workspace ID.
   * @param string $entityTypeId
   *   The entity type machine name.
   * @param string $operation
   *   The operation being performed.
   * @param string $bundle
   *   The entity bundle (optional, for nodes this would be the node type).
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The entity being accessed (for event dispatching).
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user account (for event dispatching).
   *
   * @return \Drupal\Core\Access\AccessResultInterface|null
   *   AccessResult::forbidden() if access should be blocked, null otherwise.
   */
  protected function checkWorkspaceEntityRestrictions(
    string $workspaceId,
    string $entityTypeId,
    string $operation,
    string $bundle = '',
    ?EntityInterface $entity = NULL,
    ?AccountInterface $account = NULL,
  ): ?AccessResultInterface {
    // Only apply restrictions to Live workspace and editing operations.
    if ($workspaceId !== 'live' || !in_array($operation, ['create', 'update', 'delete'])) {
      return NULL;
    }

    // Dispatch event to allow other modules to implement custom Live workspace
    // restrictions.
    if ($account) {
      // For create operations without an entity, create a temporary entity for
      // the event.
      if ($operation === 'create' && !$entity) {
        // Create a temporary entity for event dispatching.
        $entityStorage = $this->entityTypeManager->getStorage($entityTypeId);
        $entityType = $this->entityTypeManager->getDefinition($entityTypeId);
        $bundleKey = $entityType->getKey('bundle');

        // Set the bundle using the correct field name.
        $values = [];
        if ($bundleKey && $bundle) {
          $values[$bundleKey] = $bundle;
        }

        $entity = $entityStorage->create($values);
      }

      if ($entity) {
        $event = new WorkspaceAccessEvent(
              $entity,
              $operation,
              $account,
              $workspaceId,
              'live'
          );

        $this->eventDispatcher->dispatch($event, 'workspaces_access.workspace_check');

        // If any subscriber set an access result, return it.
        if ($event->getAccessResult()) {
          return $event->getAccessResult();
        }
      }
    }

    return NULL;
  }

  /**
   * Implements hook_entity_access().
   */
  #[Hook('entity_access')]
  public function entityAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResultInterface {
    // Only apply to supported entities.
    if (!$this->workspaceInfo->isEntitySupported($entity)) {
      return AccessResult::neutral();
    }

    return $this->checkEntityWorkspaceAccess($entity, $operation, $account, 'entity');
  }

  /**
   * Implements hook_node_access().
   */
  #[Hook('node_access')]
  public function nodeAccess(NodeInterface $node, $operation, AccountInterface $account): AccessResultInterface {
    // Only apply to supported operations.
    if (!in_array($operation, ['view', 'update', 'delete'])) {
      return AccessResult::neutral();
    }

    return $this->checkEntityWorkspaceAccess($node, $operation, $account, 'node');
  }

  /**
   * Check if user has permission through workspace role assignment.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   * @param \Drupal\workspaces\WorkspaceInterface|string $workspace
   *   The workspace entity or workspace ID.
   * @param string $operation
   *   The operation (view, add, edit, remove).
   *
   * @return bool
   *   TRUE if user has permission, FALSE otherwise.
   */
  private function userHasWorkspaceRolePermission(AccountInterface $account, $workspace, string $operation): bool {
    // If workspace is a string (ID), load the entity.
    if (is_string($workspace)) {
      $workspace = $this->entityTypeManager
        ->getStorage('workspace')
        ->load($workspace);

      if (!$workspace) {
        return FALSE;
      }
    }

    // Load permissions from base field.
    $fieldName = "field_workspace_roles_{$operation}";
    if (!$workspace->hasField($fieldName)) {
      return FALSE;
    }

    $field_values = $workspace->get($fieldName)->getValue();
    $assignedRoles = [];
    foreach ($field_values as $value) {
      if (isset($value['value'])) {
        $assignedRoles[] = $value['value'];
      }
    }

    if (empty($assignedRoles)) {
      return FALSE;
    }

    $userRoles = $account->getRoles();

    // Check if user has any of the assigned roles.
    foreach ($assignedRoles as $roleId) {
      if (in_array($roleId, $userRoles)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Check if user has global permission for Live workspace operations.
   */
  private function userHasLiveWorkspacePermission(AccountInterface $account, string $operation): bool {
    $permission = "workspace_live_{$operation}_content";
    return $account->hasPermission($permission);
  }

  /**
   * Common method to check workspace access for entities.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being accessed.
   * @param string $operation
   *   The operation being performed.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account.
   * @param string $hook_type
   *   The hook type ('entity' or 'node').
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  private function checkEntityWorkspaceAccess(EntityInterface $entity, string $operation, AccountInterface $account, string $hook_type): AccessResultInterface {
    // Determine workspace context.
    $hasActiveWorkspace = $this->workspaceManager->hasActiveWorkspace();
    $workspaceId = 'live';
    $activeWorkspace = NULL;

    if ($hasActiveWorkspace) {
      // Get active workspace for permission checking.
      $activeWorkspace = $this->workspaceManager->getActiveWorkspace();
      $workspaceId = $activeWorkspace->id();
    }

    // Check workspace-entity type restrictions first
    // (these override all other permissions).
    $entityTypeId = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $restrictionResult = $this->checkWorkspaceEntityRestrictions(
          $workspaceId,
          $entityTypeId,
          $operation,
          $bundle,
          $entity,
          $account
      );
    if ($restrictionResult) {
      return $restrictionResult;
    }

    // Define supported operations based on hook type.
    $supportedOperations = ($hook_type === 'entity')
        ? ['view', 'create', 'update', 'delete']
        : ['view', 'update', 'delete'];

    if (!in_array($operation, $supportedOperations)) {
      return AccessResult::neutral();
    }

    // Map operation to field name.
    $operationName = match($operation) {
      'create' => 'add',
            'update' => 'edit',
            'delete' => 'remove',
            default => $operation,
    };

    // Handle workspace-specific logic.
    if ($hasActiveWorkspace && $workspaceId !== 'live') {
      // For create and edit operations, check view permission as prerequisite.
      if (in_array($operation, ['create', 'update'])) {
        if (!$this->userHasWorkspaceRolePermission($account, $activeWorkspace, 'view')) {
          return AccessResult::forbidden(
                'User does not have view permission required for ' . $operation .
                ' operation in workspace ' . $workspaceId
            )
            ->cachePerPermissions();
        }
      }

      // Handle operations in non-Live workspace.
      if ($this->userHasWorkspaceRolePermission($account, $activeWorkspace, $operationName)) {
        // User has workspace role permission, now check basic content perms.
        $entityType = $entity->getEntityType();
        $basicPermission = $operation . ' ' . $entityType->getSingularLabel();

        if ($account->hasPermission($basicPermission) || $account->hasPermission($operation . ' any content')) {
          return AccessResult::allowed('User has workspace role and content permissions')
            ->cachePerPermissions();
        }

        return AccessResult::neutral();
      }

      // User doesn't have the specific workspace role permission.
      return AccessResult::forbidden(
            'User does not have permission to ' . $operation .
            ' content in workspace ' . $workspaceId
        )
        ->cachePerPermissions();
    }
    elseif ($workspaceId === 'live') {
      // For create and edit operations, check view permission as prerequisite.
      if (in_array($operation, ['create', 'update'])) {
        if (!$this->userHasLiveWorkspacePermission($account, 'view')) {
          return AccessResult::forbidden(
                'User does not have view permission required for ' . $operation .
                ' operation in Live workspace'
            )
            ->cachePerPermissions();
        }
      }

      // Handle Live workspace permission.
      if ($this->userHasLiveWorkspacePermission($account, $operationName)) {
        // User has Live workspace permission, now check basic content perms.
        $entityType = $entity->getEntityType();
        $basicPermission = $operation . ' ' . $entityType->getSingularLabel();

        if ($account->hasPermission($basicPermission) || $account->hasPermission($operation . ' any content')) {
          return AccessResult::allowed('User has Live workspace and content permissions')
            ->cachePerPermissions();
        }

        return AccessResult::neutral();
      }

      // User doesn't have the specific Live workspace permission.
      return AccessResult::forbidden(
            'User does not have permission to ' . $operation .
            ' content in Live workspace'
        )
        ->cachePerPermissions();
    }

    // For node hook, return neutral to allow node access modules to apply
    // restrictions.
    if ($hook_type === 'node') {
      return AccessResult::neutral();
    }

    // For other operations, return neutral.
    return AccessResult::neutral();
  }

  /**
   * Implements hook_entity_create_access().
   */
  #[Hook('entity_create_access')]
  public function entityCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL): AccessResultInterface {
    $entityTypeId = $context['entity_type_id'];

    // Don't interfere with workspace entity creation - let Drupal handle this.
    if ($entityTypeId === 'workspace') {
      return AccessResult::neutral();
    }

    // Only apply our access control to supported content entities.
    $entity_type = $this->entityTypeManager->getDefinition($entityTypeId);
    if (!$this->workspaceInfo->isEntityTypeSupported($entity_type)) {
      return AccessResult::neutral();
    }

    // Determine workspace context.
    $hasActiveWorkspace = $this->workspaceManager->hasActiveWorkspace();
    if ($hasActiveWorkspace) {
      // Get active workspace for permission checking.
      $activeWorkspace = $this->workspaceManager->getActiveWorkspace();
      $workspaceId = $activeWorkspace->id();

      // Check workspace-entity type restrictions first
      // (these override all other permissions).
      $bundle = $entity_bundle ?? '';
      $restrictionResult = $this->checkWorkspaceEntityRestrictions(
            $workspaceId,
            $entityTypeId,
            'create',
            $bundle,
            NULL,
            $account
        );
      if ($restrictionResult) {
        return $restrictionResult;
      }

      // For create operations in non-Live workspace, check view permission as
      // prerequisite.
      if (!$this->userHasWorkspaceRolePermission($account, $activeWorkspace, 'view')) {
        return AccessResult::forbidden('User does not have view permission required for create operation in workspace ' . $workspaceId)
          ->cachePerPermissions();
      }

      // Handle create operation in non-Live workspace
      // Check if user has role-based perm for this workspace and operation.
      if ($this->userHasWorkspaceRolePermission($account, $activeWorkspace, 'add')) {
        // User has workspace role perm, now check basic create permissions.
        $basicPermission = 'create ' . $entity_type->getSingularLabel();
        if ($account->hasPermission($basicPermission) || $account->hasPermission('create any content')) {
          return AccessResult::allowed('User has workspace role and create permissions')
            ->cachePerPermissions();
        }
        return AccessResult::neutral();
      }

      // User doesn't have the specific workspace role permission.
      return AccessResult::forbidden('User does not have permission to create content in workspace ' . $workspaceId)
        ->cachePerPermissions();
    }
    else {
      // We're in Live workspace.
      $workspaceId = 'live';
    }

    // Check workspace-entity type restrictions first
    // (these override all other permissions).
    $bundle = $entity_bundle ?? '';

    // For entity creation, we don't have an entity object yet, so we pass null
    // but still allow event dispatching for business logic modules.
    $restrictionResult = $this->checkWorkspaceEntityRestrictions(
          $workspaceId,
          $entityTypeId,
          'create',
          $bundle,
          NULL,
          $account
      );
    if ($restrictionResult) {
      return $restrictionResult;
    }

    // Handle Live workspace permission.
    if ($workspaceId === 'live') {
      // For create operations in Live workspace, check view permission as
      // prerequisite.
      if (!$this->userHasLiveWorkspacePermission($account, 'view')) {
        return AccessResult::forbidden('User does not have view permission required for create operation in Live workspace')
          ->cachePerPermissions();
      }

      // Check if user has global perm for Live workspace create operation.
      if ($this->userHasLiveWorkspacePermission($account, 'add')) {
        // User has Live workspace perm, now check basic create permissions.
        $basicPermission = 'create ' . $entity_type->getSingularLabel();
        if ($account->hasPermission($basicPermission) || $account->hasPermission('create any content')) {
          return AccessResult::allowed('User has Live workspace and create permissions')
            ->cachePerPermissions();
        }
        return AccessResult::neutral();
      }

      // User doesn't have Live workspace create permission.
      return AccessResult::forbidden('User does not have permission to create content in Live workspace')
        ->cachePerPermissions();
    }

    // User has workspace permission - check for basic create permission.
    $basicPermission = 'create ' . $entity_type->getSingularLabel();
    if ($account->hasPermission($basicPermission) || $account->hasPermission('create any content')) {
      return AccessResult::allowed(
            'User has workspace and create permissions'
        )->cachePerPermissions();
    }

    // User has workspace permission but lacks basic create permission
    // For media entities, also check for alternative permission names.
    if ($entityTypeId === 'media') {
      $alternativePermissions = [
        'create media',
        'create media entities',
        'create media content',
        'create media item',
      ];

      foreach ($alternativePermissions as $altPermission) {
        if ($account->hasPermission($altPermission)) {
          return AccessResult::allowed(
                'User has workspace and alternative create permissions'
            )->cachePerPermissions();
        }
      }
    }

    return AccessResult::neutral();
  }

}
