<?php

declare(strict_types=1);

namespace Drupal\workspaces_access;

use Drupal\user\Entity\Role;

/**
 * Provides dynamic permissions for workspaces access.
 */
class WorkspacesAccessPermissions {

  /**
   * Returns an array of Live workspace permissions.
   *
   * @return array
   *   An array of permissions.
   */
  public static function permissions(): array {
    $permissions = [];
    $permissions["workspace_live_view_content"] = [
      'title' => '@ Live - View content',
      'description' => 'Role can view content in workspace. Requires relevant permissions (e.g., "view article content") for operations.',
      'restrict access' => TRUE,
    ];
    $permissions["workspace_live_add_content"] = [
      'title' => '@ Live - Add content',
      'description' => 'Role can create content in workspace. Requires "View content" permission to function properly. Also requires relevant content permissions (e.g., "create article content") for operations.',
      'restrict access' => TRUE,
    ];
    $permissions["workspace_live_edit_content"] = [
      'title' => '@ Live - Edit content',
      'description' => 'Role can edit content in workspace. Requires "View content" permission to function properly. Also requires relevant content permissions (e.g., "edit article content") for operations.',
      'restrict access' => TRUE,
    ];
    $permissions["workspace_live_remove_content"] = [
      'title' => '@ Live - Remove content',
      'description' => 'Role can delete content in workspace. Requires relevant permissions (e.g., "delete article content") for operations.',
      'restrict access' => TRUE,
    ];

    return $permissions;
  }

  /**
   * Get role options for workspace permission fields.
   *
   * @return array
   *   An array of role options.
   */
  public static function getRoleOptions(): array {
    $roles = Role::loadMultiple();
    $options = [];

    foreach ($roles as $role_id => $role) {
      // Exclude anonymous role from workspace permissions.
      if ($role_id !== 'anonymous') {
        $options[$role_id] = $role->label();
      }
    }

    return $options;
  }

  /**
   * Clear permission-related caches.
   */
  public static function clearPermissionCaches(): void {
    // Clear caches and reset hook implementations to ensure permissions are
    // rediscovered.
    \Drupal::moduleHandler()->resetImplementations();

    // Clear permission-related caches.
    \Drupal::cache('discovery')->deleteAll();
    \Drupal::cache('menu')->deleteAll();

    // Invalidate permission cache tags.
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['config:workspaces_access.permissions']);
  }

}
