<?php

declare(strict_types=1);

namespace Drupal\Tests\workspaces_access\Functional;

use Drupal\Core\Access\AccessResult;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\workspaces\Entity\Workspace;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\workspaces_access\Event\WorkspaceAccessEvent;

/**
 * Tests the workspace access event system.
 *
 * @group workspaces_access
 */
class WorkspaceAccessEventTest extends BrowserTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'system',
    'workspaces',
    'workspaces_access',
    'node',
    'field',
    'field_ui',
    'filter',
    'text',
    'options',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a basic node type for testing.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Create workspaces.
    Workspace::create(['id' => 'live', 'label' => 'Live'])->save();
    Workspace::create(['id' => 'staging', 'label' => 'Staging'])->save();
  }

  /**
   * Tests that WorkspaceAccessEvent is properly constructed.
   */
  public function testWorkspaceAccessEventConstruction(): void {
    // Create test entities.
    $user = $this->createUser();
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Node',
          ]
      );
    $node->save();

    // Create event.
    $event = new WorkspaceAccessEvent($node, 'update', $user, 'staging', 'Staging');

    // Test event properties.
    $this->assertEquals($node, $event->getEntity(), 'Event should contain the correct entity');
    $this->assertEquals('update', $event->getOperation(), 'Event should contain the correct operation');
    $this->assertEquals($user, $event->getAccount(), 'Event should contain the correct user');
    $this->assertEquals('staging', $event->getWorkspaceId(), 'Event should contain the correct workspace ID');
    $this->assertEquals('Staging', $event->getWorkspaceName(), 'Event should contain the correct workspace name');
    $this->assertNull($event->getAccessResult(), 'Event should start with no access result');
  }

  /**
   * Tests setting access result on WorkspaceAccessEvent.
   */
  public function testWorkspaceAccessEventSetAccessResult(): void {
    $user = $this->createUser();
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Node',
          ]
      );
    $node->save();

    $event = new WorkspaceAccessEvent($node, 'view', $user, 'live', 'Live');

    // Test setting allowed access result.
    $allowedResult = AccessResult::allowed('Test allowed');
    $event->setAccessResult($allowedResult);
    $this->assertEquals($allowedResult, $event->getAccessResult(), 'Event should return the set access result');

    // Test setting forbidden access result.
    $forbiddenResult = AccessResult::forbidden('Test forbidden');
    $event->setAccessResult($forbiddenResult);
    $this->assertEquals($forbiddenResult, $event->getAccessResult(), 'Event should return the updated access result');
  }

  /**
   * Tests event data getters.
   */
  public function testWorkspaceAccessEventGetters(): void {
    $user = $this->createUser();
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Node',
          ]
      );
    $node->save();

    $event = new WorkspaceAccessEvent($node, 'create', $user, 'development', 'Development');

    // Test entity type ID.
    $this->assertEquals('node', $event->getEntityTypeId(), 'Should return correct entity type ID');

    // Test bundle.
    $this->assertEquals('article', $event->getBundle(), 'Should return correct bundle');

    // Test operation.
    $this->assertEquals('create', $event->getOperation(), 'Should return correct operation');
  }

  /**
   * Tests event dispatching mechanism.
   */
  public function testEventDispatchingMechanism(): void {
    // Create test data.
    $user = $this->createUser(['edit any article content']);
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Node',
          ]
      );
    $node->save();

    // Switch to staging workspace.
    $workspace = Workspace::load('staging');
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // The event should be dispatched during access checks
    // We test this by ensuring the access control still works.
    $this->setCurrentUser($user);

    // Since no custom subscribers are registered, access should be
    // neutral/forbidden depending on workspace permissions.
    $accessResult = $node->access('update', $user);
    $this->assertIsBool($accessResult, 'Access check should return a boolean');
  }

}
