<?php

declare(strict_types=1);

namespace Drupal\Tests\workspaces_access\Functional;

use Drupal\user\Entity\Role;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\workspaces\Entity\Workspace;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Tests access control for workspaces access module.
 *
 * @group workspaces_access
 */
class WorkspacesAccessTest extends BrowserTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
    'system',
    'workspaces',
    'workspaces_access',
    'node',
    'field',
    'field_ui',
    'filter',
    'text',
    'options',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a basic node type for testing.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Create workspaces (but NOT a 'live' workspace -
    // Live workspace is the absence of active workspace).
    Workspace::create(['id' => 'staging', 'label' => 'Staging'])->save();
    Workspace::create(['id' => 'development', 'label' => 'Development'])->save();
  }

  /**
   * Tests Live workspace access control without Live permissions.
   */
  public function testLiveWorkspaceWithoutPermissions(): void {
    // Create a user with basic content permissions but no Live workspace
    // permissions.
    $user = $this->createUser(
          [
            'create article content',
            'edit any article content',
            'delete any article content',
          ]
      );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // When no workspace is active, we're in the Live workspace.
    // The workspaces_access module SHOULD apply Live workspace restrictions.
    // User should NOT be able to create in Live workspace without create
    // permission.
    $entityTypeManager = \Drupal::entityTypeManager();
    $createAccess = $entityTypeManager->getAccessControlHandler('node')
      ->createAccess('article', $user, [], TRUE);
    $this->assertFalse(
          $createAccess->isAllowed(),
          'User without Live create permission should not be able to create in Live workspace'
      );

    // User should NOT be able to edit in Live workspace without edit
    // permission.
    $this->assertFalse(
          $node->access('update', $user),
          'User without Live edit permission should not be able to edit in Live workspace'
      );

    // User should NOT be able to delete in Live workspace without delete
    // permission.
    $this->assertFalse(
          $node->access('delete', $user),
          'User without Live delete permission should not be able to delete in Live workspace'
      );
  }

  /**
   * Tests Live workspace create access control with Live create permission.
   */
  public function testLiveWorkspaceCreateWithPermission(): void {
    // Create a user with Live workspace create permission, view permission,
    // and content permissions.
    $user = $this->createUser(
          [
            'workspace_live_add_content',
            'workspace_live_view_content',
            'create article content',
          ]
      );
    // Note: Permission title is 'Live - Add content' but machine name remains
    // the same.
    $this->setCurrentUser($user);

    // When no workspace is active, we're in the Live workspace.
    // Test entity creation access.
    $entityTypeManager = \Drupal::entityTypeManager();
    $accessResult = $entityTypeManager->getAccessControlHandler('node')
      ->createAccess('article', $user, [], TRUE);
    $this->assertTrue(
          $accessResult->isAllowed(),
          'User with Live create permission should be able to create in Live workspace'
      );
  }

  /**
   * Tests Live workspace edit access control with Live edit permission.
   */
  public function testLiveWorkspaceEditWithPermission(): void {
    // Create a user with Live workspace edit permission and content
    // permissions.
    $user = $this->createUser(
          [
            'workspace_live_edit_content',
            'workspace_live_view_content',
            'edit any article content',
          ]
      );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // When no workspace is active, we're in the Live workspace.
    // User SHOULD be able to edit in Live workspace with the Live edit
    // permission.
    $this->assertTrue(
          $node->access('update', $user),
          'User with Live edit permission should be able to edit in Live workspace'
      );
  }

  /**
   * Tests Live workspace delete access control with Live delete permission.
   */
  public function testLiveWorkspaceDeleteWithPermission(): void {
    // Create a user with Live workspace delete permission and content
    // permissions.
    $user = $this->createUser(
          [
            'workspace_live_remove_content',
            'delete any article content',
          ]
      );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // When no workspace is active, we're in the Live workspace.
    // User SHOULD be able to delete in Live workspace with the Live delete
    // permission.
    $this->assertTrue($node->access('delete', $user), 'User with Live delete permission should be able to delete in Live workspace');
  }

  /**
   * Tests that Live workspace permissions require content permissions.
   */
  public function testLiveWorkspaceRequiresContentPermissions(): void {
    // Create users with Live workspace permissions but no content permissions.
    $createUser = $this->createUser(
          [
            'workspace_live_add_content',
            'workspace_live_view_content',
          // No content permissions granted.
          ]
      );
    $editUser = $this->createUser(
          [
            'workspace_live_edit_content',
            'workspace_live_view_content',
          // No content permissions granted.
          ]
      );
    $deleteUser = $this->createUser(
          [
            'workspace_live_remove_content',
          // No content permissions granted.
          ]
      );

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // When no workspace is active, we're in the Live workspace.
    // Test create permission without content permissions.
    $this->setCurrentUser($createUser);
    $entityTypeManager = \Drupal::entityTypeManager();
    $createAccess = $entityTypeManager->getAccessControlHandler('node')
      ->createAccess('article', $createUser, [], TRUE);
    $this->assertFalse(
          $createAccess->isAllowed(),
          'User with Live create permission but no content permissions should not be able to create'
      );

    // Test edit permission without content permissions.
    $this->setCurrentUser($editUser);
    $this->assertFalse(
          $node->access('update', $editUser),
          'User with Live edit permission but no content permissions should not be able to edit'
      );

    // Test delete permission without content permissions.
    $this->setCurrentUser($deleteUser);
    $this->assertFalse(
          $node->access('delete', $deleteUser),
          'User with Live delete permission but no content permissions should not be able to delete'
      );
  }

  /**
   * Tests Live workspace access requires explicit permission.
   */
  public function testLiveWorkspaceRequiresExplicitPermission(): void {
    // Create a user with basic content permissions but no Live workspace
    // permission.
    $user = $this->createUser(
          [
            'create article content',
            'edit any article content',
          ]
      );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // When no workspace is active, we're in the Live workspace.
    // User should NOT be able to edit in Live workspace without explicit
    // permission.
    $this->assertFalse($node->access('update', $user), 'User without Live workspace permission should not be able to edit in Live workspace');
  }

  /**
   * Tests users without Live workspace perms cannot edit in Live workspace.
   */
  public function testNoLiveWorkspacePermissionsBlocksAccess(): void {
    // Create a user with content permissions but no Live workspace permissions.
    $user = $this->createUser(
          [
            'create article content',
            'edit any article content',
          ]
      );
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // When no workspace is active, we're in the Live workspace.
    $this->assertFalse($node->access('update', $user), 'User without Live workspace permissions should not be able to edit in Live workspace');
  }

  /**
   * Tests field-based permissions for non-Live workspaces.
   */
  public function testWorkspaceFieldPermissions(): void {
    // Create a test workspace.
    $workspace = Workspace::create(['id' => 'test_workspace', 'label' => 'Test Workspace']);
    $workspace->save();
    $workspace = Workspace::load('test_workspace');

    // Assign roles to workspace fields.
    $workspace->set('field_workspace_roles_view', [['value' => 'content_editor']]);
    $workspace->set('field_workspace_roles_edit', [['value' => 'content_editor']]);
    $workspace->save();

    // Create a user with the content_editor role and content permissions.
    $user = $this->createUser(
          [
            'access content',
            'edit any article content',
          ]
      );
    $this->setCurrentUser($user);

    // Manually assign the content_editor role to the user.
    $user->addRole('content_editor');
    $user->save();

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // Switch to test workspace.
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // User should be able to view and edit in the test workspace.
    $this->assertTrue($node->access('view', $user), 'User with workspace role should be able to view in workspace');
    $this->assertTrue($node->access('update', $user), 'User with workspace role should be able to edit in workspace');

    // User should NOT be able to delete (no delete permission assigned)
    $this->assertFalse($node->access('delete', $user), 'User without delete role should not be able to delete in workspace');
  }

  /**
   * Tests that workspace field permissions require content permissions.
   */
  public function testWorkspaceFieldPermissionsRequireContentPermissions(): void {
    // Create a test workspace.
    $workspace = Workspace::create(['id' => 'test_workspace2', 'label' => 'Test Workspace 2'])->save();
    $workspace = Workspace::load('test_workspace2');

    // Assign edit role but don't give content permissions.
    $workspace->set('field_workspace_roles_edit', [['value' => 'content_editor']]);
    $workspace->save();

    // Create a user with the role but no content permissions.
    $user = $this->createUser([], 'content_editor');
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // Switch to test workspace.
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // User should NOT be able to edit (missing content permissions)
    $this->assertFalse($node->access('update', $user), 'User with workspace role but no content permissions should not be able to edit');
  }

  /**
   * Tests workspace field creation and validation.
   */
  public function testWorkspaceFieldCreation(): void {
    // Create a workspace for testing fields (since we don't have 'live').
    $workspace = Workspace::create(['id' => 'field_test', 'label' => 'Field Test Workspace']);
    $workspace->save();
    $workspace = Workspace::load('field_test');

    // Verify that the permission fields exist.
    $this->assertTrue($workspace->hasField('field_workspace_roles_view'), 'Workspace should have view roles field');
    $this->assertTrue($workspace->hasField('field_workspace_roles_add'), 'Workspace should have add roles field');
    $this->assertTrue($workspace->hasField('field_workspace_roles_edit'), 'Workspace should have edit roles field');
    $this->assertTrue($workspace->hasField('field_workspace_roles_remove'), 'Workspace should have remove roles field');

    // Test field cardinality (should allow multiple values)
    $fieldDefinition = $workspace->getFieldDefinition('field_workspace_roles_view');
    $this->assertEquals(-1, $fieldDefinition->getFieldStorageDefinition()->getCardinality(), 'Field should allow multiple values');
  }

  /**
   * Tests multiple roles assigned to workspace fields.
   */
  public function testMultipleRolesInWorkspaceFields(): void {
    // Create a test workspace.
    $workspace = Workspace::create(['id' => 'multi_role_workspace', 'label' => 'Multi Role Workspace']);
    $workspace->save();
    $workspace = Workspace::load('multi_role_workspace');

    // Make sure the roles exist and have the required permissions.
    $roles = ['content_editor', 'senior_editor', 'other_role'];
    foreach ($roles as $role_id) {
      $role = \Drupal::entityTypeManager()->getStorage('user_role')->load($role_id);
      if (!$role) {
        $role = Role::create([
          'id' => $role_id,
          'label' => ucfirst(str_replace('_', ' ', $role_id)),
        ]);
        $role->save();
      }
      // Grant the required permission to all roles.
      $role->grantPermission('edit any article content');
      $role->save();
    }

    // Assign multiple roles to edit field.
    $workspace->set(
          'field_workspace_roles_edit', [
          ['value' => 'content_editor'],
          ['value' => 'senior_editor'],
          ]
      );
    // Also assign view permissions as prerequisite for edit operations.
    $workspace->set(
          'field_workspace_roles_view', [
          ['value' => 'content_editor'],
          ['value' => 'senior_editor'],
          ]
      );
    $workspace->save();

    // Create users with different roles.
    $editorUser = $this->createUser();
    $editorUser->addRole('content_editor');
    $editorUser->save();

    $seniorEditorUser = $this->createUser();
    $seniorEditorUser->addRole('senior_editor');
    $seniorEditorUser->save();

    $otherUser = $this->createUser();
    $otherUser->addRole('other_role');
    $otherUser->save();

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // Switch to test workspace.
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // Both assigned roles should have edit access.
    $this->setCurrentUser($editorUser);
    $this->assertTrue($node->access('update', $editorUser), 'Content editor should be able to edit');

    $this->setCurrentUser($seniorEditorUser);
    $this->assertTrue($node->access('update', $seniorEditorUser), 'Senior editor should be able to edit');

    // Unassigned role should not have access.
    $this->setCurrentUser($otherUser);
    $this->assertFalse($node->access('update', $otherUser), 'Unassigned role should not be able to edit');
  }

  /**
   * Tests event system integration for custom access control.
   */
  public function testEventSystemIntegration(): void {
    // Create a test workspace.
    $workspace = Workspace::create(['id' => 'event_test_workspace', 'label' => 'Event Test Workspace']);
    $workspace->save();
    $workspace = Workspace::load('event_test_workspace');

    // Assign role to workspace view field.
    $workspace->set('field_workspace_roles_view', [['value' => 'authenticated']]);
    $workspace->save();

    // Create a user with basic permissions.
    $user = $this->createUser(['access content']);
    $this->setCurrentUser($user);

    // Create a test node.
    $node = Node::create(
          [
            'type' => 'article',
            'title' => 'Test Article',
          ]
      );
    $node->save();

    // Switch to test workspace.
    \Drupal::service('workspaces.manager')->setActiveWorkspace($workspace);

    // Test that event is dispatched
    // (we can't easily test custom subscribers in kernel test)
    // but we can verify the event dispatching mechanism exists.
    // The event should be dispatched during access checks
    // This is tested implicitly through the access control flow.
    $this->assertTrue($node->access('view', $user), 'Basic access should work when event system is in place');
  }

  /**
   * Tests workspace permission persistence across operations.
   */
  public function testWorkspacePermissionPersistence(): void {
    // Create a test workspace.
    $workspace = Workspace::create(['id' => 'persist_test', 'label' => 'Persist Test'])->save();
    $workspace = Workspace::load('persist_test');

    // Set permissions.
    $workspace->set('field_workspace_roles_edit', [['value' => 'test_role']]);
    $workspace->save();

    // Reload workspace to ensure persistence.
    $reloadedWorkspace = Workspace::load('persist_test');

    // Verify permissions are persisted.
    $fieldValues = $reloadedWorkspace->get('field_workspace_roles_edit')->getValue();
    $this->assertCount(1, $fieldValues, 'Permission should be persisted');
    $this->assertEquals('test_role', $fieldValues[0]['value'], 'Permission value should be correct');
  }

}
