<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\EventSubscriber;

use Drupal\workspaces_access\WorkspacesAccessPermissions;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\workspaces\Event\WorkspacePostPublishEvent;
use Drupal\workspaces\Event\WorkspacePrePublishEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for workspace access management.
 */
class WorkspaceSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected CacheTagsInvalidatorInterface $cacheTagsInvalidator,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected TimeInterface $time,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      WorkspacePostPublishEvent::class => 'onWorkspacePostPublish',
      WorkspacePrePublishEvent::class => 'onWorkspacePrePublish',
    ];
  }

  /**
   * Reacts to workspace post-publish events.
   *
   * @param \Drupal\workspaces\Event\WorkspacePostPublishEvent $event
   *   The workspace post-publish event.
   */
  public function onWorkspacePostPublish(WorkspacePostPublishEvent $event): void {
    $publishedWorkspace = $event->getWorkspace();

    // Check if WSE module is enabled and workspace cloning is configured.
    if ($this->shouldCloneWorkspacePermissions($publishedWorkspace)) {
      $this->copyPermissionsToClonedWorkspace($publishedWorkspace);
    }

    // Existing cache invalidation logic.
    $this->invalidatePermissionCache();
  }

  /**
   * Reacts to workspace pre-publish events.
   *
   * @param \Drupal\workspaces\Event\WorkspacePrePublishEvent $event
   *   The workspace pre-publish event.
   */
  public function onWorkspacePrePublish(WorkspacePrePublishEvent $event): void {
    // Since we only manage Live workspace permissions now, no cache
    // invalidation needed for workspace publish events.
  }

  /**
   * Determines if workspace permissions should be cloned.
   *
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace being published.
   *
   * @return bool
   *   TRUE if permissions should be cloned, FALSE otherwise.
   */
  protected function shouldCloneWorkspacePermissions($workspace): bool {
    // Only proceed if WSE module is enabled.
    if (!$this->moduleHandler->moduleExists('wse')) {
      return FALSE;
    }

    // Check WSE global setting.
    $wse_settings = $this->configFactory->get('wse.settings');
    $clone_on_publish = $wse_settings->get('clone_on_publish') ?? FALSE;

    // Check workspace-specific override.
    if ($workspace->_clone_on_publish !== NULL) {
      $clone_on_publish = (bool) $workspace->_clone_on_publish;
    }

    return $clone_on_publish;
  }

  /**
   * Copy permissions from published workspace to newly cloned workspace.
   *
   * @param \Drupal\workspaces\WorkspaceInterface $published_workspace
   *   The workspace that was just published.
   */
  protected function copyPermissionsToClonedWorkspace($published_workspace): void {
    // Get all workspaces to find the newly cloned one.
    $workspaces = $this->entityTypeManager
      ->getStorage('workspace')
      ->loadMultiple();

    // Find workspace with same label but different ID (the clone)
    $published_id = $published_workspace->id();

    foreach ($workspaces as $workspace) {
      // Skip the published workspace itself.
      if ($workspace->id() === $published_id) {
        continue;
      }

      // Check if this is likely the cloned workspace.
      if ($this->isClonedWorkspace($published_workspace, $workspace)) {
        $this->copyWorkspacePermissions($published_workspace, $workspace);
        $workspace->save();

        // Log the permission copy operation.
        $this->loggerFactory->get('workspaces_access')
          ->info(
                  'Copied workspace permissions from @source to @target', [
                    '@source' => $published_workspace->label(),
                    '@target' => $workspace->label(),
                  ]
              );

        break;
      }
    }
  }

  /**
   * Determines if a workspace is the cloned version of another.
   *
   * @param \Drupal\workspaces\WorkspaceInterface $source
   *   The source workspace.
   * @param \Drupal\workspaces\WorkspaceInterface $candidate
   *   The candidate workspace to check.
   *
   * @return bool
   *   TRUE if candidate is a clone of source, FALSE otherwise.
   */
  protected function isClonedWorkspace($source, $candidate): bool {
    // Check if labels match (WSE creates clones with same label)
    if ($candidate->label() !== $source->label()) {
      return FALSE;
    }

    // Check if candidate has WSE status 'open' (newly cloned workspaces)
    if ($candidate->hasField('status') && $candidate->get('status')->value !== 'open') {
      return FALSE;
    }

    // Check if candidate was created recently (within last 5 minutes)
    $now = $this->time->getCurrentTime();
    $candidate_created = $candidate->getCreatedTime();
    $time_diff = $now - $candidate_created;

    // Allow 5 minute window for the cloning operation.
    return $time_diff <= 300;
  }

  /**
   * Copy workspace permissions from source to target workspace.
   *
   * @param \Drupal\workspaces\WorkspaceInterface $source
   *   The source workspace to copy permissions from.
   * @param \Drupal\workspaces\WorkspaceInterface $target
   *   The target workspace to copy permissions to.
   */
  protected function copyWorkspacePermissions($source, $target): void {
    $permission_fields = [
      'field_workspace_roles_view',
      'field_workspace_roles_add',
      'field_workspace_roles_edit',
      'field_workspace_roles_remove',
    ];

    foreach ($permission_fields as $field_name) {
      if ($source->hasField($field_name) && $target->hasField($field_name)) {
        $source_values = $source->get($field_name)->getValue();

        // Only copy if source has permissions set.
        if (!empty($source_values)) {
          $target->set($field_name, $source_values);

          $this->loggerFactory->get('workspaces_access')
            ->debug(
                    'Copied @field permissions from @source to @target', [
                      '@field' => $field_name,
                      '@source' => $source->label(),
                      '@target' => $target->label(),
                    ]
                );
        }
      }
    }
  }

  /**
   * Invalidates permission-related caches.
   */
  protected function invalidatePermissionCache(): void {
    // Clear permission caches.
    WorkspacesAccessPermissions::clearPermissionCaches();
  }

}
