<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\Controller;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Url;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\PermissionHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for workspaces access debugging.
 */
class WorkspacesAccessController extends ControllerBase {

  use StringTranslationTrait;

  /**
   * The permission handler.
   */
  protected PermissionHandlerInterface $permissionHandler;

  /**
   * The cache backend.
   */
  protected CacheBackendInterface $cache;

  /**
   * Constructs a WorkspacesAccessController object.
   */
  public function __construct(
    PermissionHandlerInterface $permissionHandler,
    CacheBackendInterface $cache,
  ) {
    $this->permissionHandler = $permissionHandler;
    $this->cache = $cache;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('user.permissions'),
      $container->get('cache.discovery')
    );
  }

  /**
   * Debug page to show workspace permissions.
   */
  public function debug() {
    $build = [];

    // Get workspace count for testing.
    $workspace_storage = $this->entityTypeManager()->getStorage('workspace');
    $workspaces = $workspace_storage->loadMultiple();
    $test_result = count($workspaces);

    $build['module_test'] = [
      '#type' => 'markup',
      '#markup' => $this->t(
          '<h3>Module Test Result: @result workspaces found</h3>', [
            '@result' => $test_result,
          ]
      ),
    ];

    // Show workspace role assignments.
    $build['workspaces'] = [
      '#type' => 'table',
      '#caption' => $this->t('Workspace Role Assignments'),
      '#header' => [
        $this->t('Workspace ID'),
        $this->t('Label'),
        $this->t('View Roles'),
        $this->t('Add Roles'),
        $this->t('Edit Roles'),
        $this->t('Remove Roles'),
      ],
      '#rows' => [],
    ];

    foreach ($workspaces as $workspace) {
      $workspace_id = $workspace->id();
      $workspace_label = $workspace->label();

      // Get role assignments for each operation.
      $view_roles = $this->getFieldRoleNames($workspace, 'field_workspace_roles_view');
      $add_roles = $this->getFieldRoleNames($workspace, 'field_workspace_roles_add');
      $edit_roles = $this->getFieldRoleNames($workspace, 'field_workspace_roles_edit');
      $remove_roles = $this->getFieldRoleNames($workspace, 'field_workspace_roles_remove');

      $build['workspaces']['#rows'][] = [
        $workspace_id,
        $workspace_label,
        implode(', ', $view_roles),
        implode(', ', $add_roles),
        implode(', ', $edit_roles),
        implode(', ', $remove_roles),
      ];
    }

    // Show field information.
    $build['fields'] = [
      '#type' => 'table',
      '#caption' => $this->t('Workspace Permission Fields'),
      '#header' => [
        $this->t('Field Name'),
        $this->t('Operation'),
        $this->t('Storage Type'),
        $this->t('Description'),
      ],
      '#rows' => [
      [
        'field_workspace_roles_view',
        $this->t('View Content'),
        $this->t('Content Field'),
        $this->t('Roles that can view content in this workspace'),
      ],
      [
        'field_workspace_roles_add',
        $this->t('Add Content'),
        $this->t('Content Field'),
        $this->t('Roles that can create content in this workspace'),
      ],
      [
        'field_workspace_roles_edit',
        $this->t('Edit Content'),
        $this->t('Content Field'),
        $this->t('Roles that can edit content in this workspace'),
      ],
      [
        'field_workspace_roles_remove',
        $this->t('Delete Content'),
        $this->t('Content Field'),
        $this->t('Roles that can delete content in this workspace'),
      ],
      ],
    ];

    // Add field test section.
    $build['field_test'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Configuration Test'),
      '#open' => TRUE,
    ];

    // Test role options function.
    $role_options = workspaces_access_get_role_options();

    $build['field_test']['content'] = [
      '#type' => 'markup',
      '#markup' => $this->t(
          '<p>Role options function returned @count roles:</p>', [
            '@count' => count($role_options),
          ]
      ),
    ];

    if (!empty($role_options)) {
      $role_list = '<ul>';
      foreach ($role_options as $role_id => $role_label) {
        $role_list .= '<li><strong>' . $role_id . ':</strong> ' . $role_label . '</li>';
      }
      $role_list .= '</ul>';
      $build['field_test']['content']['#markup'] .= $role_list;
    }

    // Add force rediscovery button.
    $build['force_rediscovery'] = [
      '#type' => 'details',
      '#title' => $this->t('Force Hook Rediscovery'),
      '#open' => TRUE,
    ];

    $build['force_rediscovery']['content'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p>If permissions are not showing up, click this button to force Drupal to rediscover hooks:</p>'),
    ];

    $build['force_rediscovery']['button'] = [
      '#type' => 'link',
      '#title' => $this->t('Force Rediscovery'),
      '#url' => Url::fromRoute('workspaces_access.force_rediscovery'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    return $build;
  }

  /**
   * Force hook rediscovery and redirect back to debug page.
   */
  public function forceRediscovery() {
    // Clear permission-related caches.
    $this->cache->deleteAll();
    $this->cache('menu')->deleteAll();

    $this->messenger()->addMessage($this->t('Hook rediscovery forced. Check the debug page and permissions.'));

    return $this->redirect('workspaces_access.debug');
  }

  /**
   * Gets the permission handler service.
   */
  protected function getPermissionHandler() {
    return $this->permissionHandler;
  }

  /**
   * Gets role names from workspace base fields.
   */
  protected function getFieldRoleNames($workspace, string $fieldName): array {
    if (!$workspace->hasField($fieldName)) {
      return [];
    }

    $field_values = $workspace->get($fieldName)->getValue();
    $role_names = [];

    foreach ($field_values as $value) {
      if (isset($value['value'])) {
        try {
          $role = $this->entityTypeManager()->getStorage('user_role')->load($value['value']);
          if ($role) {
            $role_names[] = $role->label();
          }
        }
        catch (\Exception $e) {
          // Role might not exist, skip it.
        }
      }
    }

    return $role_names;
  }

}
