<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\Event;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event for workspace access checks.
 *
 * Allows other modules to modify or extend workspace access control logic.
 */
class WorkspaceAccessEvent extends Event {

  /**
   * Constructs a WorkspaceAccessEvent.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being accessed.
   * @param string $operation
   *   The operation being performed (view, create, update, delete).
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account performing the operation.
   * @param string $workspaceId
   *   The workspace ID.
   * @param string $workspaceName
   *   The workspace name/label.
   * @param \Drupal\Core\Access\AccessResultInterface|null $accessResult
   *   The current access result (can be modified by event subscribers).
   */
  public function __construct(
    protected EntityInterface $entity,
    protected string $operation,
    protected AccountInterface $account,
    protected string $workspaceId,
    protected string $workspaceName,
    protected ?AccessResultInterface $accessResult = NULL,
  ) {
  }

  /**
   * Gets the entity being accessed.
   */
  public function getEntity(): EntityInterface {
    return $this->entity;
  }

  /**
   * Gets the operation being performed.
   */
  public function getOperation(): string {
    return $this->operation;
  }

  /**
   * Gets the user account.
   */
  public function getAccount(): AccountInterface {
    return $this->account;
  }

  /**
   * Gets the workspace ID.
   */
  public function getWorkspaceId(): string {
    return $this->workspaceId;
  }

  /**
   * Gets the workspace name.
   */
  public function getWorkspaceName(): string {
    return $this->workspaceName;
  }

  /**
   * Gets the current access result.
   */
  public function getAccessResult(): ?AccessResultInterface {
    return $this->accessResult;
  }

  /**
   * Sets the access result.
   */
  public function setAccessResult(AccessResultInterface $accessResult): void {
    $this->accessResult = $accessResult;
  }

  /**
   * Gets the entity type ID.
   */
  public function getEntityTypeId(): string {
    return $this->entity->getEntityTypeId();
  }

  /**
   * Gets the entity bundle.
   */
  public function getBundle(): string {
    return $this->entity->bundle();
  }

}
