<?php

declare(strict_types=1);

namespace Drupal\workspaces_access\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\workspaces_access\WorkspacesAccessPermissions;

/**
 * Form alteration hooks for workspace permission fields.
 */
class WorkspaceFormHooks {

  use StringTranslationTrait;

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The logger factory.
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a WorkspaceFormHooks object.
   */
  public function __construct(
    Connection $database,
    LoggerChannelFactoryInterface $loggerFactory,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->database = $database;
    $this->loggerFactory = $loggerFactory;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Target workspace add and edit forms.
    if (in_array($form_id, ['workspace_add_form', 'workspace_edit_form', 'workspace-add-form', 'workspace-edit-form'])) {
      $this->loggerFactory->get('system')->info('Workspaces Access: Form altered for ' . $form_id);
      $this->alterWorkspaceForm($form, $form_state);
      // Add validate handler.
      $form['#validate'][] = 'workspaces_access_form_validate';
    }
  }

  /**
   * Alter workspace forms to add permission fields.
   */
  private function alterWorkspaceForm(array &$form, FormStateInterface $form_state): void {
    // Add permission fields section.
    $form['workspace_permissions'] = [
      '#type' => 'details',
      '#title' => $this->t('Workspace Permissions'),
      '#description' => $this->t('Configure which user roles have permission to perform operations in this workspace.'),
      '#open' => TRUE,
      '#weight' => 50,
      '#tree' => TRUE,
    ];

    $this->addPermissionFields($form, $form_state);
    $this->populateRoleOptions($form);
  }

  /**
   * Add permission fields to the form.
   */
  private function addPermissionFields(array &$form, FormStateInterface $form_state): void {
    $operations = [
      'view' => $this->t('View content'),
      'add' => $this->t('Add content'),
      'edit' => $this->t('Edit content'),
      'remove' => $this->t('Delete content'),
    ];

    foreach ($operations as $operation => $label) {
      $fieldName = "field_workspace_roles_{$operation}";

      // Add specific descriptions for add and edit operations that require
      // view permission.
      $description = $this->t('Select roles that should have permission to @operation content in this workspace.', ['@operation' => strtolower((string) $label)]);
      if ($operation === 'add') {
        $description .= ' ' . $this->t('Note: Role must also have "View content" permission to create content.');
      }
      elseif ($operation === 'edit') {
        $description .= ' ' . $this->t('Note: Role must also have "View content" permission to edit content.');
      }

      $form['workspace_permissions'][$fieldName] = [
        '#type' => 'checkboxes',
        '#title' => $label,
        '#description' => $description,
        // Will be populated dynamically.
        '#options' => [],
        '#default_value' => $this->getFieldDefaultValue($form_state, $fieldName),
        '#weight' => array_search($operation, array_keys($operations)) + 10,
      ];
    }
  }

  /**
   * Populate role options for all permission fields.
   */
  private function populateRoleOptions(array &$form): void {
    $options = WorkspacesAccessPermissions::getRoleOptions();

    // Set options for each permission field.
    $operations = ['view', 'add', 'edit', 'remove'];
    foreach ($operations as $operation) {
      $fieldName = "field_workspace_roles_{$operation}";
      if (isset($form['workspace_permissions'][$fieldName])) {
        $form['workspace_permissions'][$fieldName]['#options'] = $options;
      }
    }
  }

  /**
   * Get default value for a field from database tables.
   */
  private function getFieldDefaultValue(FormStateInterface $form_state, string $fieldName): array {
    $form_object = $form_state->getFormObject();
    if ($form_object && method_exists($form_object, 'getEntity')) {
      $entity = $form_object->getEntity();
      if ($entity) {
        // Check if there's user input for this field (from nested structure)
        $user_input = $form_state->getUserInput();
        if (!empty($user_input)) {
          // Check for nested structure: workspace_permissions[field_name].
          if (isset($user_input['workspace_permissions'][$fieldName])) {
            return $user_input['workspace_permissions'][$fieldName];
          }
          // Check for direct field value.
          if (isset($user_input[$fieldName])) {
            return $user_input[$fieldName];
          }
        }

        // No user input, query database directly.
        $table = 'workspace__' . $fieldName;
        $query = $this->database->select($table, 'f')
          ->fields('f', [$fieldName . '_value'])
          ->condition('f.entity_id', $entity->id())
          ->condition('f.revision_id', $entity->getRevisionId())
          ->condition('f.deleted', 0)
          ->orderBy('f.delta');
        $results = $query->execute()->fetchCol();
        return $results ?: [];
      }
    }
    return [];
  }

}
