<?php

namespace Drupal\writesonic_ai_analytics\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\Core\Site\Settings;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\ServerException;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\writesonic_ai_analytics\Service\AnalyticsConfigService;

/**
 * Event subscriber for capturing and logging HTTP request analytics.
 *
 * This service intercepts HTTP requests and responses to collect analytics data
 * and send it to an external ingestion server for processing and analysis.
 */
class RequestLoggerSubscriberService implements EventSubscriberInterface {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The analytics config service.
   *
   * @var \Drupal\writesonic_ai_analytics\Service\AnalyticsConfigService
   */
  protected AnalyticsConfigService $analyticsConfigService;

  /**
   * Constructs a new RequestLoggerSubscriberService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\writesonic_ai_analytics\Service\AnalyticsConfigService $analytics_config_service
   *   The analytics config service.
   */
  public function __construct(
    ClientInterface $http_client,
    RequestStack $request_stack,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
    AnalyticsConfigService $analytics_config_service
  ) {
    $this->httpClient = $http_client;
    $this->requestStack = $request_stack;
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->logger = $logger_factory->get('writesonic_ai_analytics');
    $this->analyticsConfigService = $analytics_config_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => 'onKernelRequest',
      KernelEvents::RESPONSE => 'onKernelResponse',
    ];
  }

  /**
   * Handles kernel request events.
   *
   * Redirects to configuration page if API key is not configured.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onKernelRequest(RequestEvent $event): void {
    // Only handle main requests, not sub-requests
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();
    $current_path = $request->getPathInfo();

    if ($this->isExcludedPath($current_path)) {
      return;
    }

    // Check if API key is configured
    $api_key = $this->analyticsConfigService->getApiKey();

    // If api key is missing, redirect to the configuration form
    if (empty($api_key)) {
      // Add a message to inform the user
      $this->messenger->addWarning(
        'Writesonic AI Analytics requires configuration. Please set up your API key.'
      );

      $url = Url::fromRoute('writesonic_ai_analytics.settings');
      $response = new RedirectResponse($url->toString());
      $event->setResponse($response);
    }
  }

  /**
   * Handles kernel response events.
   *
   * Captures analytics data and sends it to the external ingestion server
   * if tracking is enabled and the request meets the criteria.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onKernelResponse(ResponseEvent $event): void {
    // Only handle main requests, not sub-requests
    if (!$event->isMainRequest()) {
      return;
    }

    // Check if tracking is enabled
    if (!$this->analyticsConfigService->isTrackingEnabled()) {
      return;
    }

    $request = $event->getRequest();
    $response = $event->getResponse();

    // Skip tracking for specific conditions
    if ($this->shouldSkipTracking($request)) {
      return;
    }

    // Collect analytics data
    $analytics_data = $this->collectAnalyticsData($request, $response);

    // Send data to external service asynchronously
    $this->sendAnalyticsDataAsync($analytics_data);
  }

  /**
   * Determines if tracking should be skipped for the current request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @return bool
   *   TRUE if tracking should be skipped, FALSE otherwise.
   */
  private function shouldSkipTracking($request): bool {
    $current_path = $request->getPathInfo();

    if ($this->isExcludedPath($current_path)) {
      return TRUE;
    }

    return FALSE;
  }


  private function isExcludedPath($path): bool {
    $excluded_paths = [
      '/admin',
      '/system',
      '/core',
      '/sites/default/files',
      '/modules',
      '/themes',
      '/contextual',
      '/quickedit',
      '/toolbar',
      '/batch',
      '/install',
      '/update',
    ];

    foreach ($excluded_paths as $excluded_path) {
      if ($this->pathStartsWith($path, $excluded_path)) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Collects analytics data from the request and response.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param \Symfony\Component\HttpFoundation\Response $response
   *   The response object.
   *
   * @return array
   *   Analytics data array.
   */
  private function collectAnalyticsData($request, $response): array {
    $ip = $this->getClientIp($request) ?: '';
    return [
      'ua' => $request->headers->get('User-Agent', ''),
      'referrer' => $request->headers->get('Referer', ''),
      'ip' => $ip,
      'country_code' => $this->getCountryCode($request),
      'url' => $request->getUri(),
      'method' => $request->getMethod(),
      'x_forwarded_for' => $request->headers->get('X-Forwarded-For', ''),
      'x_real_ip' => $request->headers->get('X-Real-IP', $ip),
      'response_status' => (string) $response->getStatusCode(),
      'integration_name' => 'drupal',
      'site_version' => \Drupal::VERSION,
      'site_title' => \Drupal::config('system.site')->get('name') ?: ''
    ];
  }

  /**
   * Gets all HTTP headers from the request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return array
   *   Array of headers.
   */
  private function getAllHeaders($request): array {
    $headers = [];
    
    foreach ($request->headers->all() as $name => $values) {
      $headers[$name] = is_array($values) ? implode(', ', $values) : $values;
    }
    
    return $headers;
  }

  /**
   * Gets the client IP address with enhanced detection.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return string
   *   The client IP address.
   */
  private function getClientIp($request): string {
    $ip_keys = [
      'HTTP_CF_CONNECTING_IP',
      'HTTP_CLIENT_IP',
      'HTTP_X_FORWARDED_FOR',
      'HTTP_X_FORWARDED',
      'HTTP_X_CLUSTER_CLIENT_IP',
      'HTTP_FORWARDED_FOR',
      'HTTP_FORWARDED',
      'REMOTE_ADDR',
    ];

    foreach ($ip_keys as $key) {
      $value = $request->server->get($key);
      if ($value && filter_var($value, FILTER_VALIDATE_IP)) {
        return $value;
      }
    }

    // Fallback to standard method
    return $request->getClientIp() ?: '';
  }

  /**
   * Gets the country code from various sources.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return string
   *   The country code.
   */
  private function getCountryCode($request): string {
    // Try Cloudflare header first
    if ($cf_country = $request->headers->get('CF-IPCountry')) {
      return $cf_country;
    }
    
    // Try other common headers
    $country_headers = [
      'X-Country-Code',
      'GeoIP-Country-Code',
      'X-Country',
    ];
    
    foreach ($country_headers as $header) {
      if ($country = $request->headers->get($header)) {
        return $country;
      }
    }
    
    return ''; 
  }

  /**
   * Sends analytics data asynchronously.
   *
   * @param array $analytics_data
   *   The analytics data to send.
   */
  private function sendAnalyticsDataAsync(array $analytics_data): void {
    // Register shutdown function to send data after response
    drupal_register_shutdown_function([$this, 'sendAnalyticsData'], $analytics_data);
  }

  /**
   * Sends analytics data to the external ingestion server.
   *
   * @param array $analytics_data
   *   The analytics data to send.
   */
  public function sendAnalyticsData(array $analytics_data): void {
    try {
      $api_key = $this->analyticsConfigService->getApiKey();
      $api_endpoint = $this->analyticsConfigService->getApiEndpoint();
      $api_endpoint_path = $this->analyticsConfigService->getApiEndpointPath();

      if (empty($api_key) || empty($api_endpoint) || empty($api_endpoint_path)) {
        $this->logger->warning('Analytics configuration incomplete. Skipping data transmission.');
        return;
      }

      $full_url = rtrim($api_endpoint, '/') . $api_endpoint_path;

      $response = $this->httpClient->post($full_url, [
        'json' => $analytics_data,
        'headers' => [
          'Content-Type' => 'application/json',
          'x-api-key' => $api_key,
        ],
        'timeout' => 5,
      ]);

      if ($response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
        $this->logger->info('Analytics data sent successfully. Status: @status', [
          '@status' => $response->getStatusCode(),
        ]);
      }

    } catch (ConnectException $e) {
      $this->logger->error('Failed to connect to analytics service: @message', [
        '@message' => $e->getMessage(),
      ]);
    } 
  }

  /**
   * Helper method to check if a path starts with a given prefix.
   *
   * @param string $path
   *   The path to check.
   * @param string $prefix
   *   The prefix to check for.
   *
   * @return bool
   *   TRUE if the path starts with the prefix, FALSE otherwise.
   */
  private function pathStartsWith(string $path, string $prefix): bool {
    return strpos($path, $prefix) === 0;
  }

}
