<?php

namespace Drupal\writesonic_ai_analytics\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Writesonic AI Analytics settings.
 *
 * Provides a secure interface for configuring the Writesonic AI analytics tracking
 * module with proper validation and error handling.
 */
class AnalyticsSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'writesonic_ai_analytics.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'writesonic_ai_analytics_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('writesonic_ai_analytics.settings');

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p>Configure analytics tracking settings for intercepting and logging HTTP requests.</p>'),
    ];

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Enter your analytics ingestion server API key. This is required for authentication.'),
      '#default_value' => $config->get('api_key'),
      '#required' => TRUE,
      '#attributes' => [
        'autocomplete' => 'off',
      ],
    ];



    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Toggle Analytics Tracking'),
      '#description' => $this->t('Enable or disable analytics data collection and transmission.'),
      '#default_value' => $config->get('enabled') ?? TRUE,
    ];


    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate API key if provided
    $api_key = $form_state->getValue('api_key');
    if (!empty($api_key)) {
      // Check minimum length
      if (strlen($api_key) < 1) {
        $form_state->setErrorByName('api_key', $this->t('API key must not be empty.'));
      }
      
      // Check for potentially unsafe characters
      if (preg_match('/[<>"\'&]/', $api_key)) {
        $form_state->setErrorByName('api_key', $this->t('API key contains invalid characters.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('writesonic_ai_analytics.settings');
    
    $config->set('api_key', $form_state->getValue('api_key'))
      ->set('enabled', $form_state->getValue('enabled'))
      ->save();

    parent::submitForm($form, $form_state);
  }

} 