<?php

namespace Drupal\writesonic_ai_analytics\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for managing analytics configuration with environment variable fallbacks.
 *
 * This service provides a secure way to access configuration values with
 * environment variable support while maintaining proper Drupal patterns.
 */
class AnalyticsConfigService {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new AnalyticsConfigService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Get API key with environment variable fallback.
   *
   * Priority order: Environment variable > Configuration > Default empty string.
   *
   * @return string
   *   The API key value.
   */
  public function getApiKey(): string {
    // Use getenv() instead of direct $_ENV access for better security
    $env_api_key = getenv('ANALYTICS_API_KEY');
    if ($env_api_key !== FALSE && !empty($env_api_key)) {
      return $env_api_key;
    }

    $config_api_key = $this->configFactory->get('writesonic_ai_analytics.settings')->get('api_key');
    return $config_api_key ?? '';
  }

  /**
   * Get API endpoint with fallback to configuration.
   *
   * @return string
   *   The API endpoint URL.
   */
  public function getApiEndpoint(): string {
    $config_endpoint = $this->configFactory->get('writesonic_ai_analytics.settings')->get('api_endpoint');
    return $config_endpoint ?? '';
  }

  /**
   * Get API endpoint path with fallback to configuration.
   *
   * @return string
   *   The API endpoint path.
   */
  public function getApiEndpointPath(): string {
    $config_endpoint_path = $this->configFactory->get('writesonic_ai_analytics.settings')->get('api_endpoint_path');
    return $config_endpoint_path ?? '';
  }

  /**
   * Check if tracking is enabled.
   *
   * Environment variable can override configuration setting.
   *
   * @return bool
   *   TRUE if tracking is enabled, FALSE otherwise.
   */
  public function isTrackingEnabled(): bool {
    // Environment variable can disable tracking
    $env_enabled = getenv('ANALYTICS_ENABLED');
    if ($env_enabled !== FALSE && $env_enabled === 'false') {
      return FALSE;
    }

    $config_enabled = $this->configFactory->get('writesonic_ai_analytics.settings')->get('enabled');
    return $config_enabled ?? TRUE;
  }

} 