<?php

namespace Drupal\wsdata\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Defines the Web Service Server entity.
 *
 * @ConfigEntityType(
 *   id = "wsserver",
 *   label = @Translation("Web Service Server"),
 *   handlers = {
 *     "list_builder" = "Drupal\wsdata\WSServerListBuilder",
 *     "form" = {
 *       "add" = "Drupal\wsdata\Form\WSServerForm",
 *       "edit" = "Drupal\wsdata\Form\WSServerForm",
 *       "delete" = "Drupal\wsdata\Form\WSServerDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\wsdata\WSServerHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "wsserver",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   links = {
 *     "add-form" = "/admin/structure/wsserver/add",
 *     "edit-form" = "/admin/structure/wsserver/{wsserver}/edit",
 *     "delete-form" = "/admin/structure/wsserver/{wsserver}/delete",
 *     "collection" = "/admin/structure/wsserver"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "endpoint",
 *     "wsconnector",
 *     "languagehandling",
 *     "settings"
 *   }
 * )
 */
class WSServer extends ConfigEntityBase {
  use StringTranslationTrait;

  /**
   * The default degraded backoff time.
   *
   * @var int
   */
  static public $WSCONFIG_DEFAULT_DEGRADED_BACKOFF = 900;

  /**
   * The Web Service Server ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The Web Service Server label.
   *
   * @var string
   */
  protected $label;

  /**
   * The Web Service Server endpoint.
   *
   * @var string
   */
  public $endpoint;

  /**
   * The Web Service Server connector.
   *
   * @var string
   */
  public $wsconnector;

  /**
   * The Web Service Server settings.
   *
   * @var array
   */
  public $settings;

  /**
   * The Web Service Server connector instance.
   *
   * @var \Drupal\wsdata\Plugin\WSConnectorBase
   */
  public $wsconnectorInst;

  /**
   * The Web Service Server state.
   *
   * @var array
   */
  public $state;

  /**
   * The Web Service Server overrides.
   *
   * @var array
   */
  public $overrides;

  /**
   * The Web Service Server language handling.
   *
   * @var string
   */
  protected $languagehandling;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $values, $entity_type) {
    parent::__construct($values, $entity_type);
    $this->state = \Drupal::state()->get('wsdata.wsserver.' . $this->id, []);
    $this->overrides = [];

    // Allow the state to override the configured endpoint.
    $this->setEndpoint($this->endpoint);

    $wsconnectorman = \Drupal::service('plugin.manager.wsconnector');
    $wscdefs = $wsconnectorman->getDefinitions();

    if (isset($wscdefs[$this->wsconnector])) {
      $this->wsconnectorInst = $wsconnectorman->createInstance($this->wsconnector);
      $this->wsconnectorInst->setEndpoint($this->endpoint);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function __destruct() {
    if (!empty($this->id)) {
      \Drupal::state()->set('wsdata.wsserver.' . $this->id, $this->state);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getMethods() {
    return $this->wsconnectorInst->getMethods();
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultMethod() {
    $methods = array_keys($this->getMethods());
    return reset($methods);
  }

  /**
   * {@inheritdoc}
   */
  public function setEndpoint($endpoint) {
    if (isset($this->state['endpoint'])) {
      $this->overrides['endpoint'] = $endpoint;
      $this->endpoint = $this->state['endpoint'];
    }
    else {
      $this->endpoint = $endpoint;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoint() {
    return $this->endpoint;
  }

  /**
   * {@inheritdoc}
   */
  public function disable($degraded = FALSE) {
    $reason = '';

    if ($degraded) {
      if (!isset($this->state['degraded_backoff'])) {
        $this->state['degraded_backoff'] = WSServer::$WSCONFIG_DEFAULT_DEGRADED_BACKOFF;
      }
      if ($this->state['degraded_backoff'] == 0) {
        return;
      }

      $reason = '  ' . $this->t('Automatically disabled due to degrated service.');
      $this->state['degraded'] = time();
    }

    $this->state['disabled'] = TRUE;
    \Drupal::logger('wsdata')->warning(
      $this->t(
        'WSServer %label (%type) was disabled.',
        [
          '%label' => $this->label(),
          '%type' => $this->wsconnector,
        ]
      ) . $reason);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function enable($degraded = FALSE) {
    unset($this->state['degraded']);
    unset($this->state['disabled']);

    $reason = '';
    if ($degraded) {
      $reason = '  ' . $this->t('Automatically re-enabling previously degrated service.');
    }

    \Drupal::logger('wsdata')->notice(
      $this->t(
        'WSConfig Type %label (%type) was enabled.',
        [
          '%label' => $this->label(),
          '%type' => $this->wsconnector,
        ]
      ) . $reason);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isDisabled() {
    if (!isset($this->state['degraded_backoff'])) {
      $this->state['degraded_backoff'] = WSServer::$WSCONFIG_DEFAULT_DEGRADED_BACKOFF;
    }

    if (isset($this->state['degraded']) and $this->state['degraded'] < time() - $this->state['degraded_backoff']) {
      $this->enable(TRUE);
      return FALSE;
    }

    return $this->state['disabled'] ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getDegraded() {
    if (!isset($this->state['degraded_backoff'])) {
      $this->state['degraded_backoff'] = WSServer::$WSCONFIG_DEFAULT_DEGRADED_BACKOFF;
    }

    if (isset($this->state['degraded'])) {
      return $this->state['degraded'] - time() + $this->state['degraded_backoff'];
    }

    return 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getConnector() {
    return $this->wsconnectorInst;
  }

}
