<?php

declare(strict_types=1);

namespace Drupal\wse_task_monitor;

/**
 * Interface for workspace task persistence operations.
 */
interface WorkspaceTaskRepositoryInterface {

  /**
   * Saves a task.
   *
   * @param \Drupal\wse_task_monitor\WorkspaceTask $task
   *   The task to save.
   */
  public function save(WorkspaceTask $task): void;

  /**
   * Finds a task by ID.
   *
   * @param string $taskId
   *   The task ID.
   *
   * @return \Drupal\wse_task_monitor\WorkspaceTask|null
   *   The task, or NULL if not found.
   */
  public function find(string $taskId): ?WorkspaceTask;

  /**
   * Deletes a task.
   *
   * @param string $taskId
   *   The task ID to delete.
   */
  public function delete(string $taskId): void;

  /**
   * Finds all tasks.
   *
   * @param string|null $workspaceId
   *   Optional workspace ID filter.
   * @param \Drupal\wse_task_monitor\TaskStatus|null $status
   *   Optional status filter.
   *
   * @return \Drupal\wse_task_monitor\WorkspaceTask[]
   *   Array of tasks indexed by task ID.
   */
  public function findAll(?string $workspaceId = NULL, ?TaskStatus $status = NULL): array;

  /**
   * Finds tasks by workspace, handler class, and status.
   *
   * @param string $workspaceId
   *   The workspace ID.
   * @param string $handlerClass
   *   The handler class.
   * @param \Drupal\wse_task_monitor\TaskStatus $status
   *   The task status.
   *
   * @return \Drupal\wse_task_monitor\WorkspaceTask[]
   *   Array of tasks indexed by task ID.
   */
  public function findByHandlerClass(string $workspaceId, string $handlerClass, TaskStatus $status): array;

  /**
   * Generates a unique task ID.
   *
   * @return string
   *   A unique task identifier.
   */
  public function generateTaskId(): string;

}
