<?php

declare(strict_types=1);

namespace Drupal\wse_task_monitor_test\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\wse_task_monitor\WorkspaceTaskRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Mock controller to simulate backup task progress.
 */
class MockBackupController extends ControllerBase {

  public function __construct(
    protected readonly WorkspaceTaskRepositoryInterface $repository,
    protected readonly FileSystemInterface $fileSystem,
    protected readonly TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get(WorkspaceTaskRepositoryInterface::class),
      $container->get('file_system'),
      $container->get('datetime.time'),
    );
  }

  /**
   * Returns backup task status in the specified format.
   *
   * @param string $task_id
   *   The task ID.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with backup status.
   */
  public function getBackupStatus(string $task_id): JsonResponse {
    // Load task from repository (read-only).
    $task = $this->repository->find($task_id);
    if (!$task) {
      return new JsonResponse(['error' => 'Task not found'], 404);
    }

    // Try to read progress from file first.
    $progress_data = $this->getProgressFromFile($task_id);

    $created_time = $task->getCreatedAt();
    $task_status = $task->getStatus();

    $progress = $progress_data['progress'];
    $message = $progress_data['message'];

    // Map task status to API status format.
    $status = 'running';
    $completed_at = NULL;

    if ($task_status->isFinished()) {
      $status = $task_status->value === 'failed' ? 'failed' : 'completed';
      $completed_at = gmdate('c', $created_time + 10);
    }
    elseif ($task_status->value === 'pending') {
      $status = 'pending';
    }

    // Generate response in the custom format.
    $response = [
      'uuid' => $task->getId(),
      'event' => 'WorkspaceBackupCreated',
      'label' => $task->getLabel(),
      'description' => $message,
      'created_at' => gmdate('c', $created_time),
      'completed_at' => $completed_at,
      'status' => $status,
      'progress' => $progress,
    ];

    return new JsonResponse($response);
  }

  /**
   * Gets progress data from file.
   *
   * @param string $task_id
   *   The task ID.
   *
   * @return array|null
   *   Progress data or NULL if file doesn't exist.
   */
  private function getProgressFromFile(string $task_id): ?array {
    $temp_dir = $this->fileSystem->getTempDirectory();
    $progress_file_path = $temp_dir . '/wse_backup_' . $task_id . '.json';

    if (!file_exists($progress_file_path)) {
      return NULL;
    }

    $content = file_get_contents($progress_file_path);
    if (!$content) {
      return NULL;
    }

    $progress_data = json_decode($content, TRUE);
    if (!$progress_data) {
      return NULL;
    }

    // Check if file is stale (older than 30 seconds).
    if (isset($progress_data['timestamp']) && ($this->time->getRequestTime() - $progress_data['timestamp'] > 30)) {
      unlink($progress_file_path);
      return NULL;
    }

    return $progress_data;
  }

}
