<?php

declare(strict_types=1);

namespace Drupal\wse_menu;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\workspaces\WorkspaceInterface;
use Drupal\workspaces\WorkspaceManagerInterface;

/**
 * Segments the menu cache by the active workspace, if available.
 */
final readonly class WseMenuCache implements CacheBackendInterface, CacheTagsInvalidatorInterface {

  /**
   * Constructs a new WseMenuCache object.
   */
  public function __construct(
    private CacheBackendInterface $inner,
    private WorkspaceManagerInterface $workspaceManager,
  ) {}

  /**
   * Adds a workspace ID cache ID prefix if there is an active workspace.
   *
   * @param string $cid
   *   The original cache ID.
   *
   * @return string
   *   The prefixed cache ID.
   */
  private function suffixCid(string $cid): string {
    $workspace = $this->workspaceManager->getActiveWorkspace();
    if ($workspace instanceof WorkspaceInterface) {
      return $cid . ':workspace:' . $workspace->id();
    }
    return $cid;
  }

  /**
   * {@inheritdoc}
   */
  public function get($cid, $allow_invalid = FALSE): mixed {
    return $this->inner->get($this->suffixCid($cid), $allow_invalid);
  }

  /**
   * {@inheritdoc}
   */
  public function getMultiple(&$cids, $allow_invalid = FALSE): array {
    $cidMap = array_map($this->suffixCid(...), array_combine($cids, $cids));
    $return = $this->inner->getMultiple($cidMap, $allow_invalid);
    $cids = array_diff($cids, array_flip($cidMap));
    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function set($cid, $data, $expire = Cache::PERMANENT, array $tags = []): void {
    $this->inner->set($this->suffixCid($cid), $data, $expire, $tags);
  }

  /**
   * {@inheritdoc}
   */
  public function setMultiple(array $items): void {
    $this->inner->setMultiple(array_reduce(array_keys($items), fn ($prefixed, $cid) => [
      ...$prefixed,
      ...[$this->suffixCid($cid) => $items[$cid]],
    ], []));
  }

  /**
   * {@inheritdoc}
   */
  public function delete($cid): void {
    $this->inner->delete($this->suffixCid($cid));
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMultiple(array $cids): void {
    $this->inner->deleteMultiple(array_map($this->suffixCid(...), $cids));
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAll(): void {
    $this->inner->deleteAll();
  }

  /**
   * {@inheritdoc}
   */
  public function invalidate($cid): void {
    $this->inner->invalidate($this->suffixCid($cid));
  }

  /**
   * {@inheritdoc}
   */
  public function invalidateMultiple(array $cids): void {
    $this->inner->invalidateMultiple(array_map($this->suffixCid(...), $cids));
  }

  /**
   * {@inheritdoc}
   */
  public function invalidateAll(): void {
    $this->inner->invalidateAll();
  }

  /**
   * {@inheritdoc}
   */
  public function garbageCollection(): void {
    $this->inner->garbageCollection();
  }

  /**
   * {@inheritdoc}
   */
  public function removeBin(): void {
    $this->inner->removeBin();
  }

  /**
   * {@inheritdoc}
   */
  public function invalidateTags(array $tags): void {
    if ($this->inner instanceof CacheTagsInvalidatorInterface) {
      $this->inner->invalidateTags($tags);
    }
    else {
      $this->inner->deleteAll();
    }
  }

}
