# WSE Parallel

## Overview

WSE Parallel is a Drupal 10.3+ and 11 submodule of the Workspace Extras (WSE) project that enables parallel editing and publishing workflows for Drupal Workspaces. This module provides conflict resolution strategies and safety guards to manage scenarios where multiple workspaces are being edited and published simultaneously.

## Features

- **Conflict Resolution Strategies**: Pluggable conflict resolution system to handle parallel workspace changes
- **Configurable Warnings**: Optional warnings when publishing workspaces that have diverged from their parent
- **Permission-Based Guards**: Bypass permission for advanced users who need to override safety mechanisms
- **Admin Interface**: Easy-to-use configuration form for selecting conflict strategies and toggling warnings
- **Edit Tracking**: Automatic tracking of which users are actively editing which content in which workspaces
- **Publish Logging**: Comprehensive logging of all publish operations with from/to revision tracking
- **Workspace History**: View historical states and publish timeline for workspaces
- **Workspace Snapshots**: Point-in-time captures for closed workspaces tracking every entity revision site-wide
- **Historical Reconstruction**: Reconstruct the complete history of the site's publishes within a margin of error
- **Activity Reports**: Visibility into parallel editing activity across workspaces
- **Status Inheritance**: Manage content status across workspace hierarchies
- **Closed Workspace Handling**: Special treatment for archived workspaces

## Key Components

### Conflict Strategy Plugin System

The module implements a plugin-based architecture for conflict resolution strategies. By default, it includes a "Last Publish Wins" strategy, but developers can create custom strategies by implementing the `ConflictStrategyInterface`.

### Configuration Options

- **Conflict Strategy**: Select how conflicts should be resolved when parallel edits occur
- **Show Publish Divergence Warning**: Toggle warnings for diverged workspace publications

### Permissions

- **Bypass WSE parallel guards**: Restricted permission allowing users to bypass parallel editing and publishing safety guards

## Architecture

The module follows Drupal best practices:

- Configuration schema for settings validation
- Plugin manager for extensible conflict strategies
- Service-based architecture for dependency injection
- Proper permission and access control integration
- Database tables for edit tracking, publish logging, and workspace snapshots
- Event subscribers for workspace publish lifecycle

## Compatibility

WSE Parallel is designed to work seamlessly with:

- Drupal Core Workspaces module
- Workspaces UI module
- WSE (Workspace Extras) base module
- WSE Menu module (optional compatibility)

## Technical Details

### Default Configuration

- **conflict_strategy**: `last_publish_wins`
- **show_publish_divergence_warning**: `TRUE`

### Database Schema

#### wse_parallel_edit_session

Tracks active editing sessions:
- `sid`: Session ID (primary key)
- `entity_type`, `entity_id`: Entity being edited
- `base_revision_id`: Starting revision
- `editing_revision_id`: Current working revision
- `workspace_id`: Workspace context
- `uid`: User performing edit
- `started`, `last_seen`: Timestamps
- `status`: Session status (active/closed)

#### wse_parallel_publish_log

Logs all publish operations:
- `plid`: Publish log ID (primary key)
- `entity_type`, `entity_id`: Published entity
- `from_revision_id`: Pre-publish revision
- `to_revision_id`: Published revision
- `workspace_id`: Source workspace
- `published`: Timestamp

### Plugin Development

Developers can create custom conflict resolution strategies by:

1. Creating a new plugin class in `src/Plugin/WseParallelConflictStrategy/`
2. Implementing `ConflictStrategyInterface`
3. Using the `@WseParallelConflictStrategy` annotation
4. Implementing the required methods:
   - `onPrePublish(array $context): ConflictResolutionResult`
   - `onPostPublish(array $context): ConflictResolutionResult`
   - `resolve(ContentEntityInterface $entity, array $context): ConflictResolutionResult`

**Example:**

```php
namespace Drupal\my_module\Plugin\WseParallelConflictStrategy;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\wse_parallel\Conflict\ConflictResolutionResult;
use Drupal\wse_parallel\Conflict\ConflictStrategyInterface;

/**
 * @WseParallelConflictStrategy(
 *   id = "my_custom_strategy",
 *   label = @Translation("My Custom Strategy"),
 *   description = @Translation("Custom conflict resolution logic."),
 *   weight = 10
 * )
 */
class MyCustomStrategy extends PluginBase implements ConflictStrategyInterface {

  public function onPrePublish(array $context): ConflictResolutionResult {
    // Validate before publish.
    return ConflictResolutionResult::success('Validation passed');
  }

  public function onPostPublish(array $context): ConflictResolutionResult {
    // Cleanup after publish.
    return ConflictResolutionResult::success('Cleanup complete');
  }

  public function resolve(ContentEntityInterface $entity, array $context): ConflictResolutionResult {
    // Custom resolution logic.
    return ConflictResolutionResult::success('Conflict resolved', 'merge');
  }

  // ... implement other required methods
}
```

## Public API

WSE Parallel provides a clean public API for other modules to integrate with parallel editing features.

### WseParallel Facade

The `WseParallel` class provides static helper methods:

#### Get Entity Parallel State

```php
use Drupal\wse_parallel\WseParallel;

// Get comprehensive parallel state for an entity.
$state = WseParallel::getEntityParallelState($node);

// Check for parallel activity.
if ($state->hasParallelActivity()) {
  \Drupal::messenger()->addWarning(
    t('This content has @count active editors and may have conflicts.', [
      '@count' => $state->getActiveSessionCount(),
    ])
  );
}

// Check for newer publishes since a specific time.
$sessionStart = 1234567890;
$state = WseParallel::getEntityParallelState($node, $sessionStart);
if ($state->hasNewerPublish()) {
  $latest = $state->getLastPublished();
  drupal_set_message(t('Newer version published at @time from workspace @ws', [
    '@time' => date('Y-m-d H:i:s', $latest['published']),
    '@ws' => $latest['workspace_id'],
  ]));
}

// Get suggested action.
switch ($state->getSuggestedAction()) {
  case 'conflict':
    // Handle conflict.
    break;
  case 'review':
    // Suggest review.
    break;
  case 'merge':
    // Suggest merge.
    break;
  case 'none':
    // No action needed.
    break;
}
```

#### ParallelStateDto Methods

```php
// Check for newer publishes.
$hasNewer = $state->hasNewerPublish(); // bool

// Get active sessions (excluding current user).
$sessions = $state->getActiveSessions(); // array
$count = $state->getActiveSessionCount(); // int
$hasActive = $state->hasActiveSessions(); // bool

// Get last publish info.
$lastPublish = $state->getLastPublished(); // array|null
$wasPublished = $state->hasBeenPublished(); // bool

// Check divergence.
$diverged = $state->hasDiverged(); // bool

// Get suggested action.
$action = $state->getSuggestedAction(); // 'none'|'review'|'merge'|'conflict'

// Overall parallel activity check.
$hasActivity = $state->hasParallelActivity(); // bool

// Get summary array.
$summary = $state->toArray(); // array
```

#### Mark Session Closed

```php
use Drupal\wse_parallel\WseParallel;

// After successfully publishing a workspace.
$publisher->publish();

// Close the user's editing session for the entity.
WseParallel::markSessionClosedOnPublish($node, \Drupal::currentUser());
```

### Integration Examples

#### Custom Workflow Integration

```php
use Drupal\wse_parallel\WseParallel;

/**
 * Implements hook_node_presave().
 */
function my_module_node_presave(NodeInterface $node) {
  // Check for parallel activity before saving.
  $state = WseParallel::getEntityParallelState($node);
  
  if ($state->getSuggestedAction() === 'conflict') {
    \Drupal::messenger()->addError(
      t('Cannot save: conflicts detected with other workspaces.')
    );
    // Optionally prevent save or trigger custom workflow.
  }
}
```

#### Approval System Integration

```php
use Drupal\wse_parallel\WseParallel;

/**
 * Check if content is safe to approve.
 */
function my_approval_system_can_approve(ContentEntityInterface $entity): bool {
  $state = WseParallel::getEntityParallelState($entity);
  
  // Don't approve if there are active editors or newer publishes.
  if ($state->hasActiveSessions() || $state->hasNewerPublish()) {
    return FALSE;
  }
  
  return TRUE;
}
```

#### Custom Dashboard Widget

```php
use Drupal\wse_parallel\WseParallel;

/**
 * Build a dashboard showing entities with parallel activity.
 */
function my_module_build_parallel_activity_dashboard() {
  $nodes = \Drupal::entityTypeManager()
    ->getStorage('node')
    ->loadMultiple();
  
  $items = [];
  foreach ($nodes as $node) {
    $state = WseParallel::getEntityParallelState($node);
    
    if ($state->hasParallelActivity()) {
      $items[] = [
        'title' => $node->label(),
        'sessions' => $state->getActiveSessionCount(),
        'action' => $state->getSuggestedAction(),
        'diverged' => $state->hasDiverged(),
      ];
    }
  }
  
  return [
    '#theme' => 'my_parallel_dashboard',
    '#items' => $items,
  ];
}
```

## Use Cases

- **Multi-team Development**: Multiple teams working on different features in parallel workspaces
- **Content Staging**: Managing concurrent content updates across different editorial workflows
- **Release Management**: Coordinating multiple workspace merges during release cycles
- **Conflict Prevention**: Detecting and resolving conflicts before they impact production
- **Audit Trail**: Complete history of all publish operations with revision tracking

## Future Enhancements

This module is designed to be extended with:

- Additional conflict resolution strategies (three-way merge, field-level merge)
- Real-time conflict detection via WebSocket/Mercure
- Workspace merge previews with diff visualization
- Automated conflict resolution workflows
- Integration with external version control systems
- REST/JSON:API endpoints for parallel state inspection
- Drush commands for session management and cleanup

---

**Note**: This module requires Drupal 10.3+ or 11 and the Workspaces module to be enabled.
