<?php

namespace Drupal\wse_parallel\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Drupal\workspaces\WorkspaceRepositoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides access control for parallel entity operations.
 */
class ParallelEntityAccess {

  /**
   * Constructs a ParallelEntityAccess object.
   *
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager.
   * @param \Drupal\workspaces\WorkspaceRepositoryInterface $workspaceRepository
   *   The workspace repository.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(protected WorkspaceManagerInterface $workspaceManager, protected WorkspaceRepositoryInterface $workspaceRepository, protected LoggerInterface $logger, protected ConfigFactoryInterface $configFactory) {}

  /**
   * Checks access for parallel entity operations.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check access for.
   * @param string $operation
   *   The operation being performed.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account to check access for.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function checkAccess(EntityInterface $entity, string $operation, AccountInterface $account): AccessResultInterface {
    // Only handle specific operations that WSE restricts.
    $restricted_operations = ['revert', 'revert revision', 'delete revision'];
    if (!in_array($operation, $restricted_operations, TRUE)) {
      return AccessResult::neutral();
    }

    // Only apply to revisionable entities.
    if (!$entity->getEntityType()->isRevisionable()) {
      return AccessResult::neutral();
    }

    // Only apply to workspace-supported entity types.
    if (!$this->workspaceManager->isEntityTypeSupported($entity->getEntityType())) {
      return AccessResult::neutral();
    }

    // Never relax restrictions for menu entities (wse_menu compatibility).
    if ($this->isMenuEntity($entity)) {
      return AccessResult::neutral();
    }

    // Get the active workspace.
    $active_workspace = $this->workspaceManager->getActiveWorkspace();
    if (!$active_workspace) {
      // Not in a workspace context, defer to default behavior.
      return AccessResult::neutral();
    }

    // Always hard-block operations on closed/archived workspaces.
    // Closed workspaces should be read-only regardless of permissions.
    if (method_exists($active_workspace, 'isClosed') && $active_workspace->isClosed()) {
      $this->logger->debug('Access denied for @operation on closed workspace (hard block)', [
        '@operation' => $operation,
      ]);
      return AccessResult::neutral();
    }

    // Workspace is not closed, proceed with normal checks.

    // Check if user has bypass permission.
    if ($account->hasPermission('bypass wse parallel guards')) {
      $this->logger->debug('Access granted for @operation on @entity_type:@entity_id via bypass permission', [
        '@operation' => $operation,
        '@entity_type' => $entity->getEntityTypeId(),
        '@entity_id' => $entity->id(),
      ]);
      return AccessResult::allowed()
        ->cachePerPermissions()
        ->addCacheableDependency($entity);
    }

    // Get the entity's tracking workspace if it's workspace-aware.
    $tracking_workspace_id = NULL;
    if ($entity instanceof ContentEntityInterface && $entity->hasField('workspace')) {
      $workspace_field = $entity->get('workspace');
      if (!$workspace_field->isEmpty()) {
        $tracking_workspace_id = $workspace_field->target_id;
      }
    }

    // If entity has no tracking workspace, defer to default behavior.
    if (!$tracking_workspace_id) {
      return AccessResult::neutral();
    }

    // Check if active workspace is in the lineage of the tracking workspace.
    if ($this->isWorkspaceInLineage($active_workspace->id(), $tracking_workspace_id)) {
      $this->logger->debug('Access granted for @operation on @entity_type:@entity_id via workspace lineage', [
        '@operation' => $operation,
        '@entity_type' => $entity->getEntityTypeId(),
        '@entity_id' => $entity->id(),
      ]);
      return AccessResult::allowed()
        ->cachePerUser()
        ->addCacheableDependency($entity)
        ->addCacheableDependency($active_workspace);
    }

    // No special access granted, defer to default behavior.
    return AccessResult::neutral();
  }

  /**
   * Checks if a workspace is in the lineage of another workspace.
   *
   * This checks if the current workspace is a descendant of the target workspace.
   *
   * @param string $current_workspace_id
   *   The current workspace ID.
   * @param string $target_workspace_id
   *   The target workspace ID.
   *
   * @return bool
   *   TRUE if current workspace is in the lineage, FALSE otherwise.
   */
  protected function isWorkspaceInLineage(string $current_workspace_id, string $target_workspace_id): bool {
    // If they're the same workspace, they're in the lineage.
    if ($current_workspace_id === $target_workspace_id) {
      return TRUE;
    }

    try {
      // Get the current workspace.
      $current_workspace = $this->workspaceRepository->load($current_workspace_id);
      if (!$current_workspace) {
        return FALSE;
      }

      // Walk up the parent chain to see if we find the target workspace.
      $max_depth = $this->configFactory->get('wse_parallel.settings')->get('workspace_lineage_max_depth') ?? 20;
      $depth = 0;

      while ($current_workspace && $depth < $max_depth) {
        // Check if current workspace has a parent field.
        if ($current_workspace->getEntityTypeId() === 'workspace' && $current_workspace->hasField('parent')) {
          $parent_field = $current_workspace->get('parent');
          if ($parent_field->isEmpty()) {
            // No parent, end of chain.
            break;
          }

          $parent_id = $parent_field->target_id;

          // Check if parent matches target.
          if ($parent_id === $target_workspace_id) {
            return TRUE;
          }

          // Move to parent workspace.
          $current_workspace = $this->workspaceRepository->load($parent_id);
          $depth++;
        }
        else {
          // No parent field, can't continue.
          break;
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error checking workspace lineage: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return FALSE;
  }

  /**
   * Checks if an entity is a menu entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if the entity is a menu entity, FALSE otherwise.
   */
  protected function isMenuEntity(EntityInterface $entity): bool {
    $menu_entity_types = ['menu', 'menu_link_content'];
    return in_array($entity->getEntityTypeId(), $menu_entity_types, TRUE);
  }

}
