<?php

namespace Drupal\wse_parallel;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\EntityOwnerInterface;
use Drupal\workspaces\WorkspaceInterface;

/**
 * Service to populate workspace view tables for closed workspaces.
 */
class ClosedWorkspaceViewBuilder {

  /**
   * Constructs a ClosedWorkspaceViewBuilder object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundleInfo
   *   The entity type bundle info service.
   */
  public function __construct(protected Connection $database, protected EntityTypeManagerInterface $entityTypeManager, protected DateFormatterInterface $dateFormatter, protected EntityTypeBundleInfoInterface $bundleInfo) {}

  /**
   * Populates the workspace changes table for closed workspaces.
   *
   * @param array &$build
   *   The build array for this workspace view.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace entity.
   */
  public function populateClosedWorkspaceTable(array &$build, WorkspaceInterface $workspace): void {
    // Query workspace_published_revisions for this workspace.
    $published_data = $this->database->select('workspace_published_revisions', 'wpr')
      ->fields('wpr', ['published_revision_ids', 'published_on'])
      ->condition('workspace_id', $workspace->id())
      ->execute()
      ->fetch();

    if (!$published_data) {
      return;
    }

    $published_revisions = json_decode($published_data->published_revision_ids, TRUE);
    if (empty($published_revisions) || !is_array($published_revisions)) {
      return;
    }

    // Update the overview to show count.
    $this->updateOverviewCounts($build, $published_revisions);

    // Build table rows from published revisions.
    $this->buildTableRows($build, $published_revisions, $workspace);
  }

  /**
   * Updates the overview section with entity counts.
   *
   * @param array &$build
   *   The build array for this workspace view.
   * @param array $published_revisions
   *   The published revisions data.
   */
  protected function updateOverviewCounts(array &$build, array $published_revisions): void {
    $changes_count = [];
    foreach ($published_revisions as $entity_type_id => $revisions) {
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        continue;
      }
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $changes_count[$entity_type_id] = $entity_type->getCountLabel(count($revisions));
    }

    if ($changes_count) {
      $build['changes']['overview']['#markup'] = implode(', ', $changes_count);
    }
  }

  /**
   * Builds table rows for published revisions.
   *
   * @param array &$build
   *   The build array for this workspace view.
   * @param array $published_revisions
   *   The published revisions data.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace entity.
   */
  protected function buildTableRows(array &$build, array $published_revisions, WorkspaceInterface $workspace): void {
    $bundle_info = $this->bundleInfo->getAllBundleInfo();

    // Check if trash module is enabled.
    $trash_enabled = \Drupal::moduleHandler()->moduleExists('trash');
    $trash_manager = $trash_enabled ? \Drupal::service('trash.manager') : NULL;

    foreach ($published_revisions as $entity_type_id => $revisions) {
      // Skip entity types that no longer exist.
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        continue;
      }

      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $storage = $this->entityTypeManager->getStorage($entity_type_id);

      // Load all revisions for this entity type.
      $revision_ids = array_keys($revisions);
      $loaded_revisions = $storage->loadMultipleRevisions($revision_ids);

      // Load all users at once.
      $revision_owners = $this->loadRevisionOwners($loaded_revisions);

      // Build table rows.
      foreach ($loaded_revisions as $revision) {
        $entity_id = $revisions[$revision->getRevisionId()];

        // Check if entity is deleted.
        $is_deleted = $trash_enabled && $trash_manager->isEntityTypeEnabled($entity_type, $revision->bundle()) && trash_entity_is_deleted($revision);

        $row = [
          '#entity' => $revision,
          'title' => $this->buildTitle($revision),
          'type' => $this->buildType($revision, $entity_type, $bundle_info),
          'changed' => $this->buildChanged($revision),
          'owner' => $this->buildAuthor($revision, $revision_owners),
        ];

        // Only add operations for non-deleted entities.
        if (!$is_deleted) {
          $row['operations'] = $this->buildOperations($workspace, $entity_type_id, $entity_id);
        }
        else {
          // Add empty operations column for deleted entities to maintain table structure.
          $row['operations'] = ['#markup' => ''];
        }

        // Highlight deleted entities if trash module is enabled.
        if ($is_deleted) {
          $row['#attributes']['style'] = 'color: #a51b00; background-color: #fcf4f2;';
        }

        $build['changes']['list'][$entity_type_id . ':' . $revision->id()] = $row;
      }
    }
  }

  /**
   * Loads all user entities for revision owners.
   *
   * @param array $revisions
   *   Array of revision entities.
   *
   * @return array
   *   Array of loaded user entities keyed by user ID.
   */
  protected function loadRevisionOwners(array $revisions): array {
    $user_ids = [];
    foreach ($revisions as $revision) {
      if ($revision instanceof EntityOwnerInterface) {
        $user_ids[$revision->getOwnerId()] = $revision->getOwnerId();
      }
    }

    if ($user_ids = array_filter($user_ids)) {
      return $this->entityTypeManager->getStorage('user')->loadMultiple($user_ids);
    }

    return [];
  }

  /**
   * Builds the title render array for a revision.
   *
   * @param \Drupal\Core\Entity\EntityInterface $revision
   *   The revision entity.
   *
   * @return array
   *   Render array for the title.
   */
  protected function buildTitle($revision): array {
    return ['#markup' => $revision->label()];
  }

  /**
   * Builds the type render array for a revision.
   *
   * @param \Drupal\Core\Entity\EntityInterface $revision
   *   The revision entity.
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param array $bundle_info
   *   All bundle info.
   *
   * @return array
   *   Render array for the type.
   */
  protected function buildType($revision, $entity_type, array $bundle_info): array {
    $entity_type_id = $revision->getEntityTypeId();

    if (!isset($bundle_info[$entity_type_id])) {
      return ['#markup' => ''];
    }

    if (count($bundle_info[$entity_type_id]) > 1) {
      return [
        '#markup' => t('@entity_type_label: @entity_bundle_label', [
          '@entity_type_label' => $entity_type->getLabel(),
          '@entity_bundle_label' => $bundle_info[$entity_type_id][$revision->bundle()]['label'] ?? '',
        ]),
      ];
    }

    return ['#markup' => $bundle_info[$entity_type_id][$revision->bundle()]['label'] ?? ''];
  }

  /**
   * Builds the changed date render array for a revision.
   *
   * @param \Drupal\Core\Entity\EntityInterface $revision
   *   The revision entity.
   *
   * @return array
   *   Render array for the changed date.
   */
  protected function buildChanged($revision): array {
    $changed = $revision instanceof EntityChangedInterface
      ? $this->dateFormatter->format($revision->getChangedTime())
      : '';

    return ['#markup' => $changed];
  }

  /**
   * Builds the author render array for a revision.
   *
   * @param \Drupal\Core\Entity\EntityInterface $revision
   *   The revision entity.
   * @param array $revision_owners
   *   Array of loaded user entities.
   *
   * @return array
   *   Render array for the author.
   */
  protected function buildAuthor($revision, array $revision_owners): array {
    if ($revision instanceof EntityOwnerInterface && isset($revision_owners[$revision->getOwnerId()])) {
      return [
        '#theme' => 'username',
        '#account' => $revision_owners[$revision->getOwnerId()],
      ];
    }

    return ['#markup' => ''];
  }

  /**
   * Builds the operations render array for a revision.
   *
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace entity.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param int|string $entity_id
   *   The entity ID.
   *
   * @return array
   *   Render array for the operations.
   */
  protected function buildOperations(WorkspaceInterface $workspace, string $entity_type_id, int|string $entity_id): array {
    // Check if the workspace history route exists for this entity type.
    // This serves as a proxy to determine if this entity type is viewable.
    $route_provider = \Drupal::service('router.route_provider');
    try {
      $route_provider->getRouteByName("entity.{$entity_type_id}.workspace_history");
    }
    catch (\Symfony\Component\Routing\Exception\RouteNotFoundException $e) {
      // Route doesn't exist, this entity type isn't viewable.
      return ['#markup' => ''];
    }

    // Check if the entity type has a view_builder handler.
    // Entity types without view builders (like menu_link_content) aren't
    // meant to be viewed as standalone entities.
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    if (!$entity_type->hasViewBuilderClass()) {
      return ['#markup' => ''];
    }

    // Exclude entity types that have a default view builder but aren't
    // designed to be viewed as standalone entities.
    $excluded_types = ['menu_link_content'];
    if (in_array($entity_type_id, $excluded_types, TRUE)) {
      return ['#markup' => ''];
    }

    return [
      '#type' => 'operations',
      '#links' => [
        'quick_look' => [
          'title' => t('Quick Look'),
          'url' => \Drupal\Core\Url::fromRoute('wse_parallel.workspace_state_view_from_workspace', [
            'workspace_id' => $workspace->id(),
            'entity_type' => $entity_type_id,
            'entity' => $entity_id,
          ]),
          'attributes' => [
            'class' => ['use-ajax'],
            'data-dialog-type' => 'modal',
            'data-dialog-options' => json_encode([
              'width' => 'auto',
            ]),
          ],
        ],
        'view_state' => [
          'title' => t('View State'),
          'url' => \Drupal\Core\Url::fromRoute('wse_parallel.workspace_state_view_from_workspace', [
            'workspace_id' => $workspace->id(),
            'entity_type' => $entity_type_id,
            'entity' => $entity_id,
          ]),
        ],
      ],
    ];
  }

}
