<?php

namespace Drupal\wse_parallel\Conflict;

/**
 * Value object representing the result of a conflict resolution operation.
 */
class ConflictResolutionResult {

  /**
   * Success status.
   *
   * @var bool
   */
  private $success;

  /**
   * Result message.
   *
   * @var string
   */
  private $message;

  /**
   * Additional metadata.
   *
   * @var array
   */
  private $metadata;

  /**
   * Action taken.
   *
   * @var string
   */
  private $action;

  /**
   * Constructs a ConflictResolutionResult.
   *
   * @param bool $success
   *   Whether the resolution was successful.
   * @param string $message
   *   A human-readable message.
   * @param string $action
   *   The action taken (e.g., 'overwrite', 'merge', 'skip').
   * @param array $metadata
   *   Additional metadata about the resolution.
   */
  public function __construct(bool $success = TRUE, string $message = '', string $action = 'none', array $metadata = []) {
    $this->success = $success;
    $this->message = $message;
    $this->action = $action;
    $this->metadata = $metadata;
  }

  /**
   * Creates a successful result.
   *
   * @param string $message
   *   Success message.
   * @param string $action
   *   The action taken.
   * @param array $metadata
   *   Additional metadata.
   *
   * @return static
   */
  public static function success(string $message = '', string $action = 'success', array $metadata = []): self {
    return new self(TRUE, $message, $action, $metadata);
  }

  /**
   * Creates a failure result.
   *
   * @param string $message
   *   Failure message.
   * @param array $metadata
   *   Additional metadata.
   *
   * @return static
   */
  public static function failure(string $message, array $metadata = []): self {
    return new self(FALSE, $message, 'failed', $metadata);
  }

  /**
   * Returns whether the resolution was successful.
   *
   * @return bool
   */
  public function isSuccess(): bool {
    return $this->success;
  }

  /**
   * Returns the result message.
   *
   * @return string
   */
  public function getMessage(): string {
    return $this->message;
  }

  /**
   * Returns the action taken.
   *
   * @return string
   */
  public function getAction(): string {
    return $this->action;
  }

  /**
   * Returns the metadata.
   *
   * @return array
   */
  public function getMetadata(): array {
    return $this->metadata;
  }

  /**
   * Gets a specific metadata value.
   *
   * @param string $key
   *   The metadata key.
   * @param mixed $default
   *   Default value if key doesn't exist.
   *
   * @return mixed
   */
  public function getMetadataValue(string $key, $default = NULL) {
    return $this->metadata[$key] ?? $default;
  }

}
