<?php

namespace Drupal\wse_parallel\Conflict;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\PluginFormInterface;

/**
 * Interface for conflict resolution strategy plugins.
 */
interface ConflictStrategyInterface extends PluginFormInterface {

  /**
   * Called before a workspace publish operation begins.
   *
   * Allows the strategy to prepare, validate, or prevent the publish.
   *
   * @param array $context
   *   Context information including:
   *   - workspace: The workspace being published.
   *   - published_revision_ids: Map of entity types to revision IDs.
   *   - entities: Array of entities being published.
   *
   * @return \Drupal\wse_parallel\Conflict\ConflictResolutionResult
   *   The result of the pre-publish check.
   */
  public function onPrePublish(array $context): ConflictResolutionResult;

  /**
   * Called after a workspace publish operation completes.
   *
   * Allows the strategy to perform cleanup or additional actions.
   *
   * @param array $context
   *   Context information including:
   *   - workspace: The workspace that was published.
   *   - published_revision_ids: Map of entity types to revision IDs.
   *   - entities: Array of entities that were published.
   *
   * @return \Drupal\wse_parallel\Conflict\ConflictResolutionResult
   *   The result of the post-publish actions.
   */
  public function onPostPublish(array $context): ConflictResolutionResult;

  /**
   * Resolves conflicts for a specific entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity with potential conflicts.
   * @param array $context
   *   Additional context information:
   *   - workspace: The workspace being published.
   *   - base_revision_id: The revision ID before publish.
   *   - workspace_revision_id: The workspace revision ID.
   *   - conflicting_changes: Array of detected conflicts.
   *
   * @return \Drupal\wse_parallel\Conflict\ConflictResolutionResult
   *   The resolution result.
   */
  public function resolve(ContentEntityInterface $entity, array $context): ConflictResolutionResult;

  /**
   * Returns a human-readable label for this strategy.
   *
   * @return string
   *   The label.
   */
  public function label(): string;

  /**
   * Returns a description of this strategy.
   *
   * @return string
   *   The description.
   */
  public function description(): string;

}
