<?php

namespace Drupal\wse_parallel\Conflict;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Manages conflict strategy plugins.
 */
class ConflictStrategyManager extends DefaultPluginManager {

  /**
   * Constructs a ConflictStrategyManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/WseParallelConflictStrategy',
      $namespaces,
      $module_handler,
      'Drupal\\wse_parallel\\Conflict\\ConflictStrategyInterface',
      'Drupal\\wse_parallel\\Annotation\\WseParallelConflictStrategy'
    );

    $this->alterInfo('wse_parallel_conflict_strategy_info');
    $this->setCacheBackend($cache_backend, 'wse_parallel_conflict_strategy_plugins');
  }

  /**
   * Gets the active conflict strategy plugin instance.
   *
   * @return \Drupal\wse_parallel\Conflict\ConflictStrategyInterface
   *   The active strategy plugin.
   */
  public function getActiveStrategy(): ConflictStrategyInterface {
    $config = \Drupal::config('wse_parallel.settings');
    $strategy_id = $config->get('conflict_strategy') ?: 'last_publish_wins';

    try {
      return $this->createInstance($strategy_id);
    }
    catch (\Exception $e) {
      // Fall back to default strategy if configured one fails.
      \Drupal::logger('wse_parallel')->error(
        'Failed to load conflict strategy @id: @message. Falling back to last_publish_wins.',
        [
          '@id' => $strategy_id,
          '@message' => $e->getMessage(),
        ]
      );
      return $this->createInstance('last_publish_wins');
    }
  }

  /**
   * Gets options for use in form select elements.
   *
   * @return array
   *   Array of strategy labels keyed by plugin ID.
   */
  public function getStrategyOptions(): array {
    $options = [];
    foreach ($this->getDefinitions() as $plugin_id => $definition) {
      $options[$plugin_id] = $definition['label'];
    }
    return $options;
  }

}
