<?php

namespace Drupal\wse_parallel\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\wse_parallel\Parallel\SessionTrackerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for the parallel activity report.
 */
class ActivityReportController extends ControllerBase {

  /**
   * Constructs an ActivityReportController object.
   *
   * @param \Drupal\wse_parallel\Parallel\SessionTrackerInterface $sessionTracker
   *   The session tracker service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   */
  public function __construct(protected SessionTrackerInterface $sessionTracker, protected DateFormatterInterface $dateFormatter) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('wse_parallel.session_tracker'),
      $container->get('date.formatter')
    );
  }

  /**
   * Displays the parallel activity report for an entity.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity
   *   The entity ID.
   *
   * @return array
   *   A render array.
   */
  public function report($entity_type, $entity) {
    // Load the entity.
    $storage = $this->entityTypeManager()->getStorage($entity_type);
    $entity_object = $storage->load($entity);

    if (!$entity_object) {
      throw new NotFoundHttpException();
    }

    $build = [];

    // Active editing sessions.
    $active_sessions = $this->sessionTracker->getActiveSessions($entity_object);

    $build['sessions'] = [
      '#type' => 'details',
      '#title' => $this->t('Active Editing Sessions (@count)', ['@count' => count($active_sessions)]),
      '#open' => TRUE,
    ];

    if (!empty($active_sessions)) {
      $session_rows = [];
      $user_storage = $this->entityTypeManager()->getStorage('user');
      $workspace_storage = $this->entityTypeManager()->getStorage('workspace');
      
      foreach ($active_sessions as $session) {
        $user = $user_storage->load($session['uid']);
        $user_name = $user ? $user->getDisplayName() : $this->t('Unknown');
        
        // Create workspace link.
        $workspace_link = $session['workspace_id'];
        try {
          $workspace = $workspace_storage->load($session['workspace_id']);
          if ($workspace && $workspace->access('view')) {
            $workspace_link = Link::createFromRoute(
              $workspace->label(),
              'entity.workspace.edit_form',
              ['workspace' => $workspace->id()]
            );
          }
        }
        catch (\Exception $e) {
          // If we can't create the link, just use the ID.
        }
        
        // Create workspace switcher button.
        $switcher_button = '';
        try {
          $workspace = $workspace_storage->load($session['workspace_id']);
          if ($workspace) {
            // Create a link to switch to this workspace and return to the entity edit form.
            $switcher_url = Url::fromRoute('entity.workspace.activate_form', [
              'workspace' => $workspace->id(),
            ], [
              'query' => [
                'destination' => $entity_object->toUrl('edit-form')->toString(),
              ],
            ]);
            
            $switcher_button = Link::fromTextAndUrl(
              $this->t('Switch & Edit'),
              $switcher_url
            )->toRenderable();
            $switcher_button['#attributes']['class'][] = 'button';
            $switcher_button['#attributes']['class'][] = 'button--small';
          }
        }
        catch (\Exception $e) {
          // If we can't create the button, leave it empty.
        }
        
        $session_rows[] = [
          $user_name,
          ['data' => $workspace_link],
          $this->t('Revision @rev', ['@rev' => $session['editing_revision_id']]),
          $this->dateFormatter->format($session['started'], 'short'),
          $this->dateFormatter->format($session['last_seen'], 'short'),
          ['data' => $switcher_button],
        ];
      }

      $build['sessions']['table'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('User'),
          $this->t('Workspace'),
          $this->t('Editing Revision'),
          $this->t('Started'),
          $this->t('Last Seen'),
          $this->t('Actions'),
        ],
        '#rows' => $session_rows,
      ];
    }
    else {
      $build['sessions']['empty'] = [
        '#markup' => '<p>' . $this->t('No active editing sessions.') . '</p>',
      ];
    }

    return $build;
  }

  /**
   * Title callback for the activity report.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity
   *   The entity ID.
   *
   * @return string
   *   The page title.
   */
  public function title($entity_type, $entity) {
    $storage = $this->entityTypeManager()->getStorage($entity_type);
    $entity_object = $storage->load($entity);

    if (!$entity_object) {
      return $this->t('Parallel Activity');
    }

    return $this->t('Parallel Activity: @label', [
      '@label' => $entity_object->label(),
    ]);
  }

}
