<?php

namespace Drupal\wse_parallel\Controller;

use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\Controller\EntityController;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\HtmlEntityFormController;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Decorator for entity form controller to relax workspace constraints.
 *
 * This decorator wraps the core entity form controller and removes the
 * EntityWorkspaceConflictConstraint for revisionable entities to enable
 * parallel editing across workspaces.
 */
class ParallelEntityFormController extends HtmlEntityFormController {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The decorated entity form controller.
   *
   * @var \Drupal\Core\Entity\HtmlEntityFormController
   */
  protected $innerController;

  /**
   * Sets the decorated controller.
   *
   * @param \Drupal\Core\Entity\HtmlEntityFormController $inner_controller
   *   The inner controller to decorate.
   */
  public function setInnerController(HtmlEntityFormController $inner_controller) {
    $this->innerController = $inner_controller;
  }

  /**
   * {@inheritdoc}
   */
  public function addForm($entity_type_id, $bundle = NULL) {
    if ($this->innerController) {
      return $this->innerController->addForm($entity_type_id, $bundle);
    }
    return parent::addForm($entity_type_id, $bundle);
  }

  /**
   * {@inheritdoc}
   */
  public function addTitle($entity_type_id) {
    if ($this->innerController) {
      return $this->innerController->addTitle($entity_type_id);
    }
    return parent::addTitle($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function editForm(EntityInterface $_entity) {
    // Get the form from the inner controller or parent.
    if ($this->innerController) {
      $form = $this->innerController->editForm($_entity);
    }
    else {
      $form = parent::editForm($_entity);
    }

    // Apply parallel editing relaxations.
    $this->relaxWorkspaceConstraints($form, $_entity);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function editTitle(RouteMatchInterface $route_match, ?EntityInterface $_entity = NULL) {
    if ($this->innerController) {
      return $this->innerController->editTitle($route_match, $_entity);
    }
    return parent::editTitle($route_match, $_entity);
  }

  /**
   * Relaxes workspace constraints for parallel editing.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being edited.
   */
  protected function relaxWorkspaceConstraints(array &$form, EntityInterface $entity) {
    // Only apply to revisionable entities.
    if (!$entity->getEntityType()->isRevisionable()) {
      return;
    }

    // Skip menu-related entities to maintain compatibility with wse_menu.
    $entity_type_id = $entity->getEntityTypeId();
    $menu_entity_types = ['menu', 'menu_link_content'];
    if (in_array($entity_type_id, $menu_entity_types, TRUE)) {
      return;
    }

    // Note: The actual relaxation of EntityWorkspaceConflictConstraint happens
    // through the validation subscriber and entity type alter hooks, not here.
    // This method primarily marks the form to indicate parallel mode is active.

    // Add a marker to the form state indicating parallel mode is active.
    if (isset($form['#form_id'])) {
      $form['#wse_parallel_relaxed'] = TRUE;
    }
  }

}
