<?php

namespace Drupal\wse_parallel\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Link;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Url;
use Drupal\wse_parallel\Publish\PublishLookup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for the workspace publish history.
 */
class WorkspaceHistoryController extends ControllerBase {

  /**
   * Constructs a WorkspaceHistoryController object.
   *
   * @param \Drupal\wse_parallel\Publish\PublishLookup $publishLookup
   *   The publish lookup service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Pager\PagerManagerInterface $pagerManager
   *   The pager manager service.
   */
  public function __construct(protected PublishLookup $publishLookup, protected DateFormatterInterface $dateFormatter, protected PagerManagerInterface $pagerManager) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('wse_parallel.publish_lookup'),
      $container->get('date.formatter'),
      $container->get('pager.manager')
    );
  }

  /**
   * Displays the workspace publish history for an entity.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $entity_type
   *   The entity type ID.
   *
   * @return array
   *   A render array.
   */
  public function history(Request $request, string $entity_type) {
    // The entity will be passed as a parameter with the entity type ID as the key.
    $entity_object = $request->attributes->get($entity_type);

    if (!$entity_object) {
      throw new NotFoundHttpException();
    }

    // Set up pager.
    $items_per_page = 50;
    $page = $request->query->get('page', 0);
    $offset = $page * $items_per_page;

    // Get total count for pager.
    $total_count = $this->publishLookup->getPublishHistoryCount($entity_type, $entity_object->id());

    // Get publish history with pager.
    $publish_history = $this->publishLookup->getPublishHistory($entity_type, $entity_object->id(), $items_per_page, $offset);

    // Show empty message if no history.
    if (empty($publish_history)) {
      return [
        'empty' => [
          '#markup' => '<p>' . $this->t('No publish history found for this @type.', [
            '@type' => $entity_object->getEntityType()->getLabel(),
          ]) . '</p>',
        ],
      ];
    }

    // Build table rows.
    $publish_rows = [];
    $workspace_storage = $this->entityTypeManager()->getStorage('workspace');

    foreach ($publish_history as $publish) {
      // Create workspace link.
      $workspace_link = $publish['workspace_id'];
      try {
        $workspace = $workspace_storage->load($publish['workspace_id']);
        if ($workspace && $workspace->access('view')) {
          $workspace_link = Link::createFromRoute(
            $workspace->label(),
            'entity.workspace.edit_form',
            ['workspace' => $workspace->id()]
          );
        }
      }
      catch (\Exception $e) {
        // If we can't create the link, just use the ID.
      }

      $publish_rows[] = [
        ['data' => $workspace_link],
        $this->t('@from → @to', [
          '@from' => $publish['from_revision_id'] ?: $this->t('Unknown'),
          '@to' => $publish['to_revision_id'],
        ]),
        $this->dateFormatter->format($publish['published'], 'medium'),
        $publish['langcode'] ?: $this->t('N/A'),
        [
          'data' => Link::fromTextAndUrl(
            $this->t('View State'),
            Url::fromRoute('wse_parallel.workspace_state_view', [
              'workspace_id' => $publish['workspace_id'],
              'entity_type' => $entity_type,
              'entity' => $entity_object->id(),
            ])
          ),
        ],
      ];
    }

    // Build the render array.
    $build = [];

    $build['history'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['workspace-history-wrapper'],
      ],
    ];

    $build['history']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Workspace'),
        $this->t('Revision Change'),
        $this->t('Published'),
        $this->t('Language'),
        $this->t('Actions'),
      ],
      '#rows' => $publish_rows,
    ];

    // Add pager.
    $this->pagerManager->createPager($total_count, $items_per_page);
    $build['history']['pager'] = [
      '#type' => 'pager',
    ];

    return $build;
  }

  /**
   * Title callback for the workspace history.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $entity_type
   *   The entity type ID.
   *
   * @return string
   *   The page title.
   */
  public function title(Request $request, string $entity_type) {
    $entity_object = $request->attributes->get($entity_type);

    if (!$entity_object) {
      return $this->t('Workspace History');
    }

    return $this->t('Workspace History: @label', [
      '@label' => $entity_object->label(),
    ]);
  }

}
