<?php

namespace Drupal\wse_parallel\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\wse_parallel\WorkspaceState\WorkspaceStateContextInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller to view entities as they were when a workspace was published.
 */
class WorkspaceStateController extends ControllerBase {

  /**
   * Constructs a WorkspaceStateController object.
   *
   * @param \Drupal\wse_parallel\WorkspaceState\WorkspaceStateContextInterface $stateContext
   *   The workspace state context service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(protected WorkspaceStateContextInterface $stateContext, protected Connection $database) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('wse_parallel.workspace_state_context'),
      $container->get('database')
    );
  }

  /**
   * Displays an entity as it was when a workspace was published.
   *
   * @param string $workspace_id
   *   The workspace ID.
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity
   *   The entity ID.
   *
   * @return array
   *   A render array.
   */
  public function view(string $workspace_id, string $entity_type, int|string $entity) {
    // Load the workspace to get its label.
    $workspace_storage = $this->entityTypeManager()->getStorage('workspace');
    $workspace_entity = $workspace_storage->load($workspace_id);

    if (!$workspace_entity) {
      throw new NotFoundHttpException();
    }

    // Get the publish timestamp from workspace_published_revisions table.
    $published_timestamp = $this->database->select('workspace_published_revisions', 'wpr')
      ->fields('wpr', ['published_on'])
      ->condition('workspace_id', $workspace_id)
      ->execute()
      ->fetchField();

    if (!$published_timestamp) {
      throw new NotFoundHttpException('This workspace has not been published yet.');
    }

    // Get the entity view.
    $build = $this->viewEntityState($workspace_id, $entity_type, $entity, $published_timestamp);

    // Create back link to workspace history.
    $back_link = Link::fromTextAndUrl(
      $this->t('← Back to Workspace History'),
      Url::fromRoute("entity.$entity_type.workspace_history", [
        $entity_type => $entity,
      ])
    )->toString();

    // Add a message at the top explaining this is a historic view.
    $build['#prefix'] = '<div class="messages messages--warning">' .
      $back_link . '<br>' .
      $this->t('You are viewing this @type as it was when workspace "@workspace" was published on @date.', [
        '@type' => $build['#entity']->getEntityType()->getLabel(),
        '@workspace' => $workspace_entity->label(),
        '@date' => \Drupal::service('date.formatter')->format($published_timestamp, 'medium'),
      ]) .
      '</div>';

    return $build;
  }

  /**
   * Displays an entity from a closed workspace manage page.
   *
   * @param string $workspace_id
   *   The workspace ID.
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity
   *   The entity ID.
   *
   * @return array
   *   A render array.
   */
  public function viewFromWorkspace(string $workspace_id, string $entity_type, int|string $entity) {
    // Get the publish timestamp from workspace_published_revisions table.
    $published_timestamp = $this->database->select('workspace_published_revisions', 'wpr')
      ->fields('wpr', ['published_on'])
      ->condition('workspace_id', $workspace_id)
      ->execute()
      ->fetchField();

    if (!$published_timestamp) {
      throw new NotFoundHttpException('This workspace has not been published yet.');
    }

    // Get the entity view without any navigation chrome.
    return $this->viewEntityState($workspace_id, $entity_type, $entity, $published_timestamp);
  }

  /**
   * Loads and displays an entity at a specific workspace publish time.
   *
   * @param string $workspace_id
   *   The workspace ID.
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity
   *   The entity ID.
   * @param int $published_timestamp
   *   The workspace publish timestamp.
   *
   * @return array
   *   A render array containing the entity view.
   */
  protected function viewEntityState(string $workspace_id, string $entity_type, int|string $entity, int $published_timestamp) {
    // Set the workspace state context so hook_entity_preload can intercept.
    $this->stateContext->setContext($workspace_id, $published_timestamp);

    try {
      // Load the entity - our hook_entity_preload will intercept and load
      // the correct historic revision.
      $storage = $this->entityTypeManager()->getStorage($entity_type);
      $entity_object = $storage->load($entity);

      if (!$entity_object) {
        throw new NotFoundHttpException();
      }

      // Build the entity view.
      $view_builder = $this->entityTypeManager()->getViewBuilder($entity_type);
      $build = $view_builder->view($entity_object, 'full');

      // Store the entity object for use by calling methods.
      $build['#entity'] = $entity_object;

      return $build;
    }
    finally {
      // Always clear the context when done.
      $this->stateContext->clearContext();
    }
  }

  /**
   * Title callback for the workspace state view.
   *
   * @param string $workspace_id
   *   The workspace ID.
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity
   *   The entity ID.
   *
   * @return string
   *   The page title.
   */
  public function title(string $workspace_id, string $entity_type, int|string $entity) {
    $storage = $this->entityTypeManager()->getStorage($entity_type);
    $entity_object = $storage->load($entity);

    if (!$entity_object) {
      return $this->t('Historic View');
    }

    return $this->t('@label (Historic View)', [
      '@label' => $entity_object->label(),
    ]);
  }

}
