<?php

namespace Drupal\wse_parallel;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\wse_parallel\Parallel\LiveRevisionResolverInterface;
use Drupal\wse_parallel\Parallel\StatusInheritanceService;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class for reacting to entity events.
 *
 * @internal
 */
class EntityOperations implements ContainerInjectionInterface {

  /**
   * Constructs a new EntityOperations instance.
   *
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager service.
   * @param \Drupal\wse_parallel\Parallel\LiveRevisionResolverInterface $liveRevisionResolver
   *   The live revision resolver service.
   * @param \Drupal\wse_parallel\Parallel\StatusInheritanceService $statusInheritance
   *   The status inheritance service.
   */
  public function __construct(protected WorkspaceManagerInterface $workspaceManager, protected LiveRevisionResolverInterface $liveRevisionResolver, protected StatusInheritanceService $statusInheritance) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('workspaces.manager'),
      $container->get('wse_parallel.live_revision_resolver'),
      $container->get('wse_parallel.status_inheritance')
    );
  }

  /**
   * Acts on an entity before it is created or updated.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being saved.
   *
   * @see hook_entity_presave()
   */
  public function entityPresave(EntityInterface $entity) {
    // Only process content entities.
    if (!$entity instanceof ContentEntityInterface) {
      return;
    }

    // Skip if entity is new - there's no live revision to compare against.
    if ($entity->isNew()) {
      return;
    }

    // Skip if entity is syncing (e.g., during workspace publish).
    if ($entity->isSyncing()) {
      return;
    }

    // Only process if workspaces are enabled for this entity type.
    if (!$this->workspaceManager->isEntityTypeSupported($entity->getEntityType())) {
      return;
    }

    // Only process if in a workspace.
    if (!$this->workspaceManager->hasActiveWorkspace()) {
      return;
    }

    $workspace = $this->workspaceManager->getActiveWorkspace();

    // Get the live revision ID as the base.
    $base_revision_id = $this->liveRevisionResolver->getLiveRevisionId($entity);

    // Check if this entity should inherit published status when forked from
    // an unpublished initial revision.
    $this->statusInheritance->checkAndApplyStatusInheritance($entity, $workspace, $base_revision_id);
  }

}
