<?php

namespace Drupal\wse_parallel\EventSubscriber;

use Drupal\Core\Entity\ContentEntityFormInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\layout_builder\Form\OverridesSectionStorageForm;
use Drupal\wse_parallel\Parallel\SessionTrackerInterface;
use Drupal\wse_parallel\Parallel\ParallelWarningBuilder;
use Drupal\workspaces\WorkspaceManagerInterface;

/**
 * Provides form alter functionality for tracking parallel editing sessions.
 *
 * Note: This class does not implement EventSubscriberInterface because it
 * works through hook_form_alter(), not through event subscription.
 */
class FormAlterSubscriber {

  use StringTranslationTrait;

  /**
   * Constructs a FormAlterSubscriber object.
   *
   * @param \Drupal\wse_parallel\Parallel\SessionTrackerInterface $sessionTracker
   *   The session tracker service.
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\wse_parallel\Parallel\ParallelWarningBuilder $warningBuilder
   *   The parallel warning builder service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(protected SessionTrackerInterface $sessionTracker, protected WorkspaceManagerInterface $workspaceManager, protected AccountProxyInterface $currentUser, protected ParallelWarningBuilder $warningBuilder, protected MessengerInterface $messenger) {}

  /**
   * Alters entity edit forms to track sessions.
   *
   * This method should be called from hook_form_alter().
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   */
  public function formAlter(array &$form, FormStateInterface $form_state, $form_id) {
    // Check if we're in a workspace.
    $active_workspace = $this->workspaceManager->getActiveWorkspace();
    if (!$active_workspace) {
      return;
    }

    // Skip for anonymous users.
    if ($this->currentUser->isAnonymous()) {
      return;
    }

    $entity = NULL;
    $is_layout_builder = FALSE;

    // Handle standard entity edit forms.
    $form_object = $form_state->getFormObject();
    if ($form_object instanceof ContentEntityFormInterface) {
      $entity = $form_object->getEntity();
    }
    // Handle Layout Builder override forms.
    elseif ($form_object instanceof OverridesSectionStorageForm) {
      $is_layout_builder = TRUE;
      // Extract entity from Layout Builder section storage.
      $section_storage = $form_state->get('section_storage');
      if ($section_storage && method_exists($section_storage, 'getContextValue')) {
        try {
          $entity = $section_storage->getContextValue('entity');
        }
        catch (\Exception $e) {
          // Context not available, skip tracking.
          return;
        }
      }
    }

    // Ensure we have a valid content entity.
    if (!$entity instanceof ContentEntityInterface) {
      return;
    }

    // Additional validation: ensure entity has valid type and ID.
    // This catches cases where entity proxies or broken objects pass instanceof checks.
    try {
      $entity_type_id = $entity->getEntityTypeId();
      $entity_id = $entity->id();

      if (empty($entity_type_id) || empty($entity_id)) {
        \Drupal::logger('wse_parallel')->warning('Skipping warning display for entity with invalid type or ID: type=@type, id=@id, class=@class', [
          '@type' => var_export($entity_type_id, TRUE),
          '@id' => var_export($entity_id, TRUE),
          '@class' => get_class($entity),
        ]);
        return;
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('wse_parallel')->error('Failed to validate entity for warning display: @message', [
        '@message' => $e->getMessage(),
      ]);
      return;
    }

    // Skip non-revisionable entities.
    if (!$entity->getEntityType()->isRevisionable()) {
      return;
    }

    // Skip new entities that haven't been saved yet.
    if ($entity->isNew()) {
      return;
    }

    // Get the current revision ID as the base revision.
    $base_revision_id = $entity->getRevisionId();

    // Check for divergence and display warnings.
    $this->checkForDivergence($form, $entity, $base_revision_id, $active_workspace);
  }

  /**
   * Checks for divergence and adds warnings to the form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity being edited.
   * @param int $base_revision_id
   *   The base revision ID.
   * @param \Drupal\workspaces\WorkspaceInterface $active_workspace
   *   The current active workspace.
   */
  protected function checkForDivergence(array &$form, ContentEntityInterface $entity, $base_revision_id, $active_workspace) {
    // Use the warning builder service to get warnings.
    $warnings = $this->warningBuilder->buildWarnings($entity, $active_workspace, $base_revision_id);

    if (empty($warnings)) {
      return;
    }

    // Display warnings using messenger.
    foreach ($warnings as $warning) {
      $this->messenger->addWarning($warning);
    }

    // Attach the CSS library only when warnings are displayed.
    $form['#attached']['library'][] = 'wse_parallel/parallel_warnings';
  }

}
