<?php

namespace Drupal\wse_parallel\EventSubscriber;

use Drupal\wse_parallel\Conflict\ConflictStrategyManager;
use Drupal\wse_parallel\Publish\PublishLoggerInterface;
use Drupal\wse_parallel\Snapshot\SnapshotManagerInterface;
use Drupal\workspaces\Event\WorkspacePrePublishEvent;
use Drupal\workspaces\Event\WorkspacePostPublishEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to workspace publish events to log revision changes.
 */
class PublishSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a PublishSubscriber object.
   *
   * @param \Drupal\wse_parallel\Publish\PublishLoggerInterface $logger
   *   The publish logger service.
   * @param \Drupal\wse_parallel\Conflict\ConflictStrategyManager $conflictStrategyManager
   *   The conflict strategy manager.
   * @param \Drupal\wse_parallel\Snapshot\SnapshotManagerInterface $snapshotManager
   *   The snapshot manager service.
   */
  public function __construct(protected PublishLoggerInterface $logger, protected ConflictStrategyManager $conflictStrategyManager, protected SnapshotManagerInterface $snapshotManager) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      WorkspacePrePublishEvent::class => ['onPrePublish', 0],
      WorkspacePostPublishEvent::class => ['onPostPublish', 0],
    ];
  }

  /**
   * Handles the pre-publish event.
   *
   * @param \Drupal\workspaces\Event\WorkspacePrePublishEvent $event
   *   The pre-publish event.
   */
  public function onPrePublish(WorkspacePrePublishEvent $event): void {
    // Log the pre-publish state.
    $this->logger->recordPrePublish(
      $event->getWorkspace(),
      $event->getPublishedRevisionIds()
    );

    // Execute conflict strategy pre-publish hook.
    try {
      $strategy = $this->conflictStrategyManager->getActiveStrategy();
      $context = [
        'workspace' => $event->getWorkspace(),
        'published_revision_ids' => $event->getPublishedRevisionIds(),
      ];

      $result = $strategy->onPrePublish($context);

      if (!$result->isSuccess()) {
        \Drupal::logger('wse_parallel')->warning(
          'Conflict strategy pre-publish check failed: @message',
          ['@message' => $result->getMessage()]
        );
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('wse_parallel')->error(
        'Error executing conflict strategy pre-publish: @message',
        ['@message' => $e->getMessage()]
      );
    }
  }

  /**
   * Handles the post-publish event.
   *
   * @param \Drupal\workspaces\Event\WorkspacePostPublishEvent $event
   *   The post-publish event.
   */
  public function onPostPublish(WorkspacePostPublishEvent $event): void {
    // Log the post-publish state.
    $this->logger->recordPostPublish(
      $event->getWorkspace(),
      $event->getPublishedRevisionIds()
    );

    // Execute conflict strategy post-publish hook.
    try {
      $strategy = $this->conflictStrategyManager->getActiveStrategy();
      $context = [
        'workspace' => $event->getWorkspace(),
        'published_revision_ids' => $event->getPublishedRevisionIds(),
      ];

      $result = $strategy->onPostPublish($context);

      if ($result->isSuccess()) {
        \Drupal::logger('wse_parallel')->info(
          'Conflict strategy post-publish completed: @message',
          ['@message' => $result->getMessage()]
        );
      }
      else {
        \Drupal::logger('wse_parallel')->warning(
          'Conflict strategy post-publish failed: @message',
          ['@message' => $result->getMessage()]
        );
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('wse_parallel')->error(
        'Error executing conflict strategy post-publish: @message',
        ['@message' => $e->getMessage()]
      );
    }

    // Create snapshot for time machine functionality after all other actions.
    $this->snapshotManager->createSnapshot(
      $event->getWorkspace(),
      $event->getPublishedRevisionIds()
    );
  }

}
