<?php

namespace Drupal\wse_parallel\EventSubscriber;

use Drupal\wse_parallel\Parallel\SessionTrackerInterface;
use Drupal\wse_parallel\Snapshot\SnapshotManagerInterface;
use Drupal\workspaces\Event\WorkspacePostPublishEvent;
use Drupal\wse\Event\WorkspaceEvents;
use Drupal\wse\Event\WorkspaceRevertEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to workspace lifecycle events to clean up session tracking.
 */
class WorkspaceSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a WorkspaceSubscriber object.
   *
   * @param \Drupal\workspaces\WorkspaceAssociationInterface $workspace_association
   *   The workspace association service.
   * @param \Drupal\wse_parallel\Parallel\SessionTrackerInterface $session_tracker
   *   The session tracker service.
   * @param \Drupal\wse_parallel\Snapshot\SnapshotManagerInterface $snapshot_manager
   *   The snapshot manager service.
   */
  public function __construct(protected $workspaceAssociation, protected SessionTrackerInterface $sessionTracker, protected SnapshotManagerInterface $snapshotManager) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      WorkspaceEvents::WORKSPACE_PRE_REVERT => ['onPreRevert', 100],
      WorkspacePostPublishEvent::class => ['onPostPublish', -100],
      WorkspaceEvents::WORKSPACE_POST_REVERT => ['onPostRevert', 0],
    ];
  }

  /**
   * Handles workspace post-publish event to clean up sessions.
   *
   * When a workspace is published, all editing sessions in that workspace
   * are complete and should be removed from tracking.
   *
   * @param \Drupal\workspaces\Event\WorkspacePostPublishEvent $event
   *   The post-publish event.
   */
  public function onPostPublish(WorkspacePostPublishEvent $event): void {
    $workspace = $event->getWorkspace();
    $workspace_id = $workspace->id();

    // Delete all sessions for this workspace since the work is complete.
    $deleted_count = $this->sessionTracker->deleteSessionsForWorkspace($workspace_id);

    if ($deleted_count > 0) {
      \Drupal::logger('wse_parallel')->info(
        'Deleted @count editing session(s) for published workspace @workspace.',
        [
          '@count' => $deleted_count,
          '@workspace' => $workspace_id,
        ]
      );
    }
  }

  /**
   * Handles workspace pre-revert event to disable parallel editing bypass.
   *
   * During revert operations, the parallel editing decorator must be disabled
   * to ensure accurate workspace tracking information is used.
   *
   * @param \Drupal\wse\Event\WorkspaceRevertEvent $event
   *   The pre-revert event.
   */
  public function onPreRevert(WorkspaceRevertEvent $event): void {
    // Tell the workspace association we're starting a revert operation.
    if (method_exists($this->workspaceAssociation, 'setReverting')) {
      $this->workspaceAssociation->setReverting(TRUE);
    }
  }

  /**
   * Handles workspace post-revert event to delete the snapshot.
   *
   * When a workspace is successfully reverted, its snapshot should be deleted
   * so that if the workspace is published again later, a fresh snapshot will
   * be created.
   *
   * @param \Drupal\wse\Event\WorkspaceRevertEvent $event
   *   The post-revert event.
   */
  public function onPostRevert(WorkspaceRevertEvent $event): void {
    $workspace = $event->getWorkspace();
    $workspace_id = $workspace->id();

    // Tell the workspace association the revert operation is complete.
    if (method_exists($this->workspaceAssociation, 'setReverting')) {
      $this->workspaceAssociation->setReverting(FALSE);
    }

    // Delete the snapshot for this workspace.
    $this->snapshotManager->deleteSnapshot($workspace_id);

    \Drupal::logger('wse_parallel')->info(
      'Deleted snapshot for workspace @workspace after successful revert.',
      ['@workspace' => $workspace_id]
    );
  }

}
