<?php

namespace Drupal\wse_parallel\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\workspaces\WorkspaceOperationFactory;

/**
 * Helper service for altering the workspace publish form.
 */
class PublishFormHelper {

  use StringTranslationTrait;

  /**
   * Constructs a PublishFormHelper object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\workspaces\WorkspaceOperationFactory $workspaceOperationFactory
   *   The workspace operation factory.
   */
  public function __construct(protected Connection $database, protected EntityTypeManagerInterface $entityTypeManager, protected MessengerInterface $messenger, protected WorkspaceOperationFactory $workspaceOperationFactory) {}

  /**
   * Alters the workspace publish form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function alterPublishForm(array &$form, FormStateInterface $form_state): void {
    // Get the workspace being published.
    $workspace = $form_state->getBuildInfo()['args'][0] ?? NULL;
    if (!$workspace) {
      return;
    }

    $workspace_id = $workspace->id();

    // Get the revisions being published.
    $workspace_publisher = $this->workspaceOperationFactory->getPublisher($workspace);
    $source_rev_diff = $workspace_publisher->getDifferringRevisionIdsOnSource();

    if (empty($source_rev_diff)) {
      return;
    }

    // Get unpublished parallel edits (entities being edited in other workspaces).
    $unpublished_parallel = $this->findUnpublishedParallelEdits($source_rev_diff, $workspace_id);

    // Show warning message if there are parallel edits.
    if (!empty($unpublished_parallel)) {
      $this->messenger->addWarning($this->t('Warning: Some content in this workspace has active editing sessions in other workspaces.'));
    }

    // Add section for entities with active sessions (not yet published).
    if (!empty($unpublished_parallel)) {
      $form['unpublished_parallel'] = [
        '#type' => 'details',
        '#title' => $this->t('Content being edited in other workspaces (unpublished)'),
        '#open' => TRUE,
        '#weight' => -50,
      ];

      $form['unpublished_parallel']['table'] = $this->buildUnpublishedTable($unpublished_parallel);
    }

    // Add section for content published since fork.
    $diverged_content = $this->findDivergedContent($source_rev_diff, $workspace_id);

    if (!empty($diverged_content)) {
      $form['diverged_content'] = [
        '#type' => 'details',
        '#title' => $this->t('Content published by other workspaces since your fork'),
        '#open' => TRUE,
        '#weight' => -40,
      ];

      $form['diverged_content']['table'] = $this->buildDivergedTable($diverged_content);
    }
  }

  /**
   * Finds unpublished parallel edits using efficient batched queries.
   *
   * @param array $source_rev_diff
   *   The source revision differences.
   * @param string $current_workspace_id
   *   The current workspace ID.
   *
   * @return array
   *   Array of entities with unpublished parallel edits.
   */
  protected function findUnpublishedParallelEdits(array $source_rev_diff, string $current_workspace_id): array {
    $unpublished = [];

    // Process each entity type separately with WHERE IN queries.
    foreach ($source_rev_diff as $entity_type_id => $entity_revisions) {
      $entity_ids = array_values($entity_revisions);

      if (empty($entity_ids)) {
        continue;
      }

      // Single query per entity type using WHERE IN.
      $sessions = $this->database->select('wse_parallel_edit_session', 's')
        ->fields('s')
        ->condition('entity_type', $entity_type_id)
        ->condition('entity_id', $entity_ids, 'IN')
        ->condition('workspace_id', $current_workspace_id, '!=')
        ->execute()
        ->fetchAll(\PDO::FETCH_ASSOC);

      // Group sessions by entity.
      foreach ($sessions as $session) {
        $key = $entity_type_id . ':' . $session['entity_id'];
        if (!isset($unpublished[$key])) {
          $unpublished[$key] = [
            'entity_type' => $entity_type_id,
            'entity_id' => $session['entity_id'],
            'sessions' => [],
          ];
        }
        $unpublished[$key]['sessions'][] = $session;
      }
    }

    return $unpublished;
  }

  /**
   * Finds content published since the workspace fork using efficient queries.
   *
   * @param array $source_rev_diff
   *   The source revision differences.
   * @param string $current_workspace_id
   *   The current workspace ID.
   *
   * @return array
   *   Array of diverged content.
   */
  protected function findDivergedContent(array $source_rev_diff, string $current_workspace_id): array {
    $diverged = [];

    // Process each entity type separately.
    foreach ($source_rev_diff as $entity_type_id => $entity_revisions) {
      $entity_ids = array_values($entity_revisions);

      if (empty($entity_ids)) {
        continue;
      }

      // Get base revisions for all entities in this workspace (single query).
      $base_revisions = $this->database->select('wse_parallel_edit_session', 's')
        ->fields('s', ['entity_id', 'base_revision_id'])
        ->condition('entity_type', $entity_type_id)
        ->condition('entity_id', $entity_ids, 'IN')
        ->condition('workspace_id', $current_workspace_id)
        ->execute()
        ->fetchAllKeyed();

      if (empty($base_revisions)) {
        continue;
      }

      // For each entity, check for publishes with from_revision_id >= base.
      foreach ($base_revisions as $entity_id => $base_revision) {
        // Find publishes where from_revision_id >= base_revision.
        $publishes = $this->database->select('wse_parallel_publish_log', 'p')
          ->fields('p')
          ->condition('entity_type', $entity_type_id)
          ->condition('entity_id', $entity_id)
          ->condition('from_revision_id', $base_revision, '>=')
          ->orderBy('published', 'DESC')
          ->execute()
          ->fetchAll(\PDO::FETCH_ASSOC);

        if (!empty($publishes)) {
          $diverged[$entity_type_id . ':' . $entity_id] = [
            'entity_type' => $entity_type_id,
            'entity_id' => $entity_id,
            'base_revision' => $base_revision,
            'publishes' => $publishes,
          ];
        }
      }
    }

    return $diverged;
  }

  /**
   * Builds the unpublished parallel edits table.
   *
   * @param array $unpublished
   *   The unpublished parallel edits data.
   *
   * @return array
   *   A render array for the table.
   */
  protected function buildUnpublishedTable(array $unpublished): array {
    $rows = [];
    $workspace_storage = $this->entityTypeManager->getStorage('workspace');
    $user_storage = $this->entityTypeManager->getStorage('user');

    foreach ($unpublished as $item) {
      $entity_type = $item['entity_type'];
      $entity_id = $item['entity_id'];

      // Load the entity.
      $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
      $entity_label = $entity ? $entity->label() : $this->t('Unknown');

      // Load the entity storage to get revision data.
      $storage = $this->entityTypeManager->getStorage($entity_type);

      foreach ($item['sessions'] as $session) {
        $workspace = $workspace_storage->load($session['workspace_id']);
        $workspace_label = $workspace ? $workspace->label() : $session['workspace_id'];

        // Create workspace link.
        $workspace_link = $workspace_label;
        if ($workspace && $workspace->access('view')) {
          $workspace_link = \Drupal\Core\Link::createFromRoute(
            $workspace_label,
            'entity.workspace.canonical',
            ['workspace' => $workspace->id()]
          );
        }

        $user = $user_storage->load($session['uid']);
        $user_name = $user ? $user->getDisplayName() : $this->t('Unknown');

        // Get the last edit time from the revision.
        $last_edit = $this->t('Unknown');
        try {
          $revision = $storage->loadRevision($session['editing_revision_id']);
          if ($revision && $revision->hasField('changed')) {
            $changed_time = $revision->get('changed')->value;
            $last_edit = \Drupal::service('date.formatter')->format($changed_time, 'short');
          }
        }
        catch (\Exception $e) {
          // If we can't load the revision, keep the default.
        }

        $rows[] = [
          $entity_label,
          $entity_type,
          $workspace_link,
          $user_name,
          $this->t('Rev @rev', ['@rev' => $session['editing_revision_id']]),
          $last_edit,
        ];
      }
    }

    return [
      '#type' => 'table',
      '#header' => [
        $this->t('Content'),
        $this->t('Type'),
        $this->t('Workspace'),
        $this->t('Revision Author'),
        $this->t('Editing Revision'),
        $this->t('Last Edit'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No unpublished parallel edits found.'),
    ];
  }

  /**
   * Builds the diverged content table.
   *
   * @param array $diverged
   *   The diverged content data.
   *
   * @return array
   *   A render array for the table.
   */
  protected function buildDivergedTable(array $diverged): array {
    $rows = [];
    $workspace_storage = $this->entityTypeManager->getStorage('workspace');
    $user_storage = $this->entityTypeManager->getStorage('user');

    foreach ($diverged as $item) {
      $entity_type = $item['entity_type'];
      $entity_id = $item['entity_id'];

      // Load the entity.
      $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
      $entity_label = $entity ? $entity->label() : $this->t('Unknown');

      // Load the entity storage to get revision authors.
      $storage = $this->entityTypeManager->getStorage($entity_type);

      foreach ($item['publishes'] as $publish) {
        $workspace = $workspace_storage->load($publish['workspace_id']);
        $workspace_label = $workspace ? $workspace->label() : $publish['workspace_id'];

        // Create workspace link.
        $workspace_link = $workspace_label;
        if ($workspace && $workspace->access('view')) {
          $workspace_link = \Drupal\Core\Link::createFromRoute(
            $workspace_label,
            'entity.workspace.canonical',
            ['workspace' => $workspace->id()]
          );
        }

        // Get the revision author.
        $revision_author = $this->t('Unknown');
        try {
          $revision = $storage->loadRevision($publish['to_revision_id']);
          if ($revision && method_exists($revision, 'getRevisionUser')) {
            $author = $revision->getRevisionUser();
            if ($author) {
              $revision_author = $author->getDisplayName();
            }
          }
        }
        catch (\Exception $e) {
          // If we can't load the revision, keep the default.
        }

        $published_time = \Drupal::service('date.formatter')->format($publish['published'], 'short');

        $rows[] = [
          $entity_label,
          $entity_type,
          $workspace_link,
          $revision_author,
          $this->t('@from → @to', [
            '@from' => $publish['from_revision_id'],
            '@to' => $publish['to_revision_id'],
          ]),
          $published_time,
        ];
      }
    }

    return [
      '#type' => 'table',
      '#header' => [
        $this->t('Content'),
        $this->t('Type'),
        $this->t('Workspace'),
        $this->t('Revision Author'),
        $this->t('Revision Change'),
        $this->t('Published'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No diverged content found.'),
    ];
  }

}
