<?php

namespace Drupal\wse_parallel\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\wse_parallel\Conflict\ConflictStrategyManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure WSE Parallel settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\wse_parallel\Conflict\ConflictStrategyManager $conflictStrategyManager
   *   The conflict strategy plugin manager.
   */
  public function __construct(protected ConflictStrategyManager $conflictStrategyManager) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('wse_parallel.conflict_strategy_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['wse_parallel.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'wse_parallel_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('wse_parallel.settings');

    // Get available conflict strategy plugins.
    $strategy_options = $this->conflictStrategyManager->getStrategyOptions();

    $form['conflict_strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('Conflict strategy'),
      '#description' => $this->t('Select the strategy to use when conflicts occur during parallel editing or publishing.'),
      '#options' => $strategy_options,
      '#default_value' => $config->get('conflict_strategy') ?: 'last_publish_wins',
      '#required' => TRUE,
    ];

    // Add descriptions for each strategy.
    $form['strategy_descriptions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['strategy-descriptions']],
    ];

    foreach ($this->conflictStrategyManager->getDefinitions() as $plugin_id => $definition) {
      $form['strategy_descriptions'][$plugin_id] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['strategy-description'],
          'data-strategy-id' => $plugin_id,
        ],
      ];

      $form['strategy_descriptions'][$plugin_id]['description'] = [
        '#markup' => '<strong>' . $definition['label'] . ':</strong> ' . $definition['description'],
      ];
    }

    $form['show_publish_divergence_warning'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show publish divergence warning'),
      '#description' => $this->t('Display a warning when publishing workspaces that have diverged from their parent.'),
      '#default_value' => $config->get('show_publish_divergence_warning') ?? TRUE,
    ];

    $form['workspace_lineage_max_depth'] = [
      '#type' => 'number',
      '#title' => $this->t('Workspace lineage max depth'),
      '#description' => $this->t('Maximum depth to traverse when checking workspace parent chains. Prevents infinite loops in case of circular references. Default is 20.'),
      '#default_value' => $config->get('workspace_lineage_max_depth') ?? 20,
      '#min' => 1,
      '#max' => 100,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('wse_parallel.settings')
      ->set('conflict_strategy', $form_state->getValue('conflict_strategy'))
      ->set('show_publish_divergence_warning', $form_state->getValue('show_publish_divergence_warning'))
      ->set('workspace_lineage_max_depth', $form_state->getValue('workspace_lineage_max_depth'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
