<?php

namespace Drupal\wse_parallel\Parallel;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for resolving live workspace revision IDs.
 */
class LiveRevisionResolver implements LiveRevisionResolverInterface {

  /**
   * Constructs a LiveRevisionResolver object.
   *
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(protected WorkspaceManagerInterface $workspaceManager, protected EntityTypeManagerInterface $entityTypeManager, protected LoggerInterface $logger) {}

  /**
   * {@inheritdoc}
   */
  public function getLiveRevisionId(ContentEntityInterface $entity): ?int {
    // If we're already in live workspace, just return the current revision ID.
    if (!$this->workspaceManager->hasActiveWorkspace()) {
      return $entity->getRevisionId();
    }

    // Store the current workspace.
    $original_workspace = $this->workspaceManager->getActiveWorkspace();

    try {
      // Switch to live workspace.
      $this->workspaceManager->switchToLive();

      // Reload the entity in live context.
      $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());
      $live_entity = $storage->load($entity->id());

      if (!$live_entity) {
        // Entity doesn't exist in live yet (new entity).
        return NULL;
      }

      // Get the live revision ID.
      return $live_entity->getRevisionId();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get live revision ID for @type:@id: @message', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
    finally {
      // Always restore the original workspace.
      if ($original_workspace) {
        $this->workspaceManager->setActiveWorkspace($original_workspace);
      }
    }
  }

}
