<?php

namespace Drupal\wse_parallel\Parallel;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\wse_parallel\Publish\PublishLookup;
use Drupal\workspaces\WorkspaceInterface;

/**
 * Service for building parallel edit warning messages.
 */
class ParallelWarningBuilder {

  use StringTranslationTrait;

  /**
   * Constructs a ParallelWarningBuilder object.
   *
   * @param \Drupal\wse_parallel\Parallel\SessionTrackerInterface $sessionTracker
   *   The session tracker service.
   * @param \Drupal\wse_parallel\Publish\PublishLookup $publishLookup
   *   The publish lookup service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(protected SessionTrackerInterface $sessionTracker, protected PublishLookup $publishLookup, protected AccountProxyInterface $currentUser, protected EntityTypeManagerInterface $entityTypeManager) {}

  /**
   * Builds warning messages for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check for warnings.
   * @param \Drupal\workspaces\WorkspaceInterface $active_workspace
   *   The current active workspace.
   * @param int $base_revision_id
   *   The base revision ID.
   * @param bool $include_link
   *   Whether to include the activity report link in the warnings.
   *
   * @return array
   *   An array of warning messages (renderable markup), or empty array if no warnings.
   */
  public function buildWarnings(ContentEntityInterface $entity, WorkspaceInterface $active_workspace, $base_revision_id, $include_link = TRUE) {
    // Check if warnings are enabled.
    $config = \Drupal::config('wse_parallel.settings');
    if (!$config->get('show_publish_divergence_warning')) {
      return [];
    }

    $warnings = [];

    // Get active sessions for this entity.
    $active_sessions = $this->sessionTracker->getActiveSessions($entity);

    // Filter sessions to find parallel editing.
    $current_workspace_id = $active_workspace->id();
    $current_user_id = $this->currentUser->id();

    $other_sessions = array_filter($active_sessions, function ($session) use ($current_workspace_id, $current_user_id) {
      return $session['workspace_id'] != $current_workspace_id || $session['uid'] != $current_user_id;
    });

    // Check for publishes since session started.
    $entity_type = $entity->getEntityTypeId();
    $entity_id = $entity->id();

    // Get the latest publish.
    $latest_publish = $this->publishLookup->latestPublish($entity_type, $entity_id);

    $has_divergence = FALSE;
    $divergence_info = [];

    // Check for divergence by comparing revision IDs.
    if ($latest_publish && $latest_publish['to_revision_id'] != $base_revision_id) {
      $has_divergence = TRUE;
      $divergence_info = $latest_publish;
    }

    // Build warning messages.
    if ($has_divergence) {
      $published_time = \Drupal::service('date.formatter')->format(
        $divergence_info['published'],
        'medium'
      );

      $warnings[] = $this->t(
        'Another workspace (@workspace) published a newer revision of this content at @time. You are working from revision @base_rev; the latest published is revision @to_rev. Your changes may conflict with the published version.',
        [
          '@workspace' => $divergence_info['workspace_id'],
          '@time' => $published_time,
          '@base_rev' => $base_revision_id,
          '@to_rev' => $divergence_info['to_revision_id'],
        ]
      );
    }

    if (!empty($other_sessions)) {
      $session_count = count($other_sessions);

      // Count sessions by workspace.
      $workspaces = [];
      foreach ($other_sessions as $session) {
        $workspaces[$session['workspace_id']] = TRUE;
      }
      $workspace_count = count($workspaces);

      if ($workspace_count > 1) {
        $warnings[] = $this->t(
          'This content is being edited in @count other workspace(s) simultaneously. Your changes may conflict.',
          ['@count' => $workspace_count]
        );
      }
      else {
        $warnings[] = $this->t(
          'This content is being edited in another workspace simultaneously. Your changes may conflict.'
        );
      }
    }

    // Add activity report link if requested and warnings exist.
    if ($include_link && !empty($warnings)) {
      $activity_url = $this->getActivityReportUrl($entity);
      if ($activity_url) {
        $link = \Drupal\Core\Link::fromTextAndUrl(
          $this->t('View parallel activity report'),
          $activity_url
        );
        $link = $link->toRenderable();
        $link['#attributes']['class'][] = 'button';
        $link['#attributes']['class'][] = 'button--small';
        $link['#attributes']['class'][] = 'use-ajax';
        $link['#attributes']['data-dialog-type'] = 'modal';
        $link['#attributes']['data-dialog-options'] = json_encode([
          'width' => 'auto',
        ]);

        // Append the button to the last warning message.
        $last_warning = array_pop($warnings);
        $combined_message = [
          '#type' => 'inline_template',
          '#template' => '{{ warning }} {{ link }}',
          '#context' => [
            'warning' => $last_warning,
            'link' => $link,
          ],
        ];
        $warnings[] = \Drupal\Core\Render\Markup::create(\Drupal::service('renderer')->renderPlain($combined_message));
      }
    }

    return $warnings;
  }

  /**
   * Gets the activity report link for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return \Drupal\Core\Url|null
   *   The URL to the activity report, or NULL if not available.
   */
  public function getActivityReportUrl(ContentEntityInterface $entity) {
    try {
      $route_provider = \Drupal::service('router.route_provider');
      if ($route_provider->getRouteByName('wse_parallel.activity_report')) {
        return Url::fromRoute('wse_parallel.activity_report', [
          'entity_type' => $entity->getEntityTypeId(),
          'entity' => $entity->id(),
        ]);
      }
    }
    catch (\Exception $e) {
      // Route may not exist.
    }

    return NULL;
  }

}
