<?php

namespace Drupal\wse_parallel\Parallel;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\workspaces\WorkspaceInformationInterface;
use Drupal\workspaces\WorkspaceInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for tracking parallel editing sessions.
 */
class SessionTracker implements SessionTrackerInterface {

  /**
   * Minimum interval between last_seen updates (in seconds).
   *
   * @var int
   */
  protected const UPDATE_THROTTLE_SECONDS = 60;

  /**
   * Constructs a SessionTracker object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(protected Connection $database, protected LoggerInterface $logger, protected WorkspaceInformationInterface $workspaceInformation) {}

  /**
   * {@inheritdoc}
   */
  public function startOrUpdateSession(ContentEntityInterface $entity, ?int $baseRevisionId, ?int $editingRevisionId, WorkspaceInterface $workspace, AccountInterface $account): int {
    // Skip anonymous users.
    if ($account->isAnonymous()) {
      return 0;
    }

    // Skip non-revisionable entities.
    if (!$entity->getEntityType()->isRevisionable()) {
      return 0;
    }

    // Skip entities that are not supported by workspaces.
    if (!$this->workspaceInformation->isEntityTypeSupported($entity->getEntityType())) {
      return 0;
    }

    // Skip menu entities for wse_menu compatibility.
    if ($this->isMenuEntity($entity)) {
      return 0;
    }

    // Skip entities without valid IDs (new/unsaved entities).
    $entity_id = $entity->id();
    if (empty($entity_id)) {
      return 0;
    }

    $entity_type = $entity->getEntityTypeId();
    $langcode = $entity->language()->getId();
    $workspace_id = $workspace->id();
    $uid = $account->id();

    // Additional validation: ensure we have valid data.
    if (empty($entity_type) || empty($workspace_id)) {
      $this->logger->warning('Skipping session tracking due to invalid data: entity_type=@type, workspace_id=@workspace, entity_id=@entity_id, langcode=@langcode, uid=@uid', [
        '@type' => var_export($entity_type, TRUE),
        '@workspace' => var_export($workspace_id, TRUE),
        '@entity_id' => var_export($entity_id, TRUE),
        '@langcode' => var_export($langcode, TRUE),
        '@uid' => var_export($uid, TRUE),
      ]);
      return 0;
    }

    // Validate langcode is actually a language code (not a number).
    if (!is_string($langcode) || empty($langcode) || is_numeric($langcode)) {
      $this->logger->warning('Skipping session tracking due to invalid langcode: entity_type=@type, entity_id=@entity_id, langcode=@langcode, workspace=@workspace', [
        '@type' => $entity_type,
        '@entity_id' => $entity_id,
        '@langcode' => var_export($langcode, TRUE),
        '@workspace' => $workspace_id,
      ]);
      return 0;
    }

    $now = \Drupal::time()->getRequestTime();

    // Use the entity's revision ID if editing revision not explicitly provided.
    if ($editingRevisionId === NULL && $entity->getRevisionId()) {
      $editingRevisionId = $entity->getRevisionId();
    }

    // Use editing revision as base if base not provided.
    if ($baseRevisionId === NULL) {
      $baseRevisionId = $editingRevisionId ?? 0;
    }

    // Check if we need to throttle the update.
    // First, try to get the existing session's last_seen timestamp.
    $existing_last_seen = $this->database->select('wse_parallel_edit_session', 's')
      ->fields('s', ['last_seen'])
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->condition('workspace_id', $workspace_id)
      ->condition('uid', $uid)
      ->condition('langcode', $langcode)
      ->execute()
      ->fetchField();

    // If session exists and throttle time hasn't passed, skip the update.
    if ($existing_last_seen !== FALSE) {
      $time_since_update = $now - $existing_last_seen;
      if ($time_since_update < self::UPDATE_THROTTLE_SECONDS) {
        // Return a placeholder SID since we're not updating.
        // We can't easily get the actual SID without another query.
        return 1;
      }
    }

    // Use MERGE operation to atomically insert or update.
    // The merge key includes langcode to ensure separate sessions per translation.
    try {
      $merge = $this->database->merge('wse_parallel_edit_session')
        ->keys([
          'entity_type' => $entity_type,
          'entity_id' => $entity_id,
          'workspace_id' => $workspace_id,
          'uid' => $uid,
          'langcode' => $langcode,
        ])
        ->fields([
          'base_revision_id' => $baseRevisionId ?? 0,
          'editing_revision_id' => $editingRevisionId ?? 0,
          'started' => $now,
          'last_seen' => $now,
        ])
        ->execute();

      // Merge returns MergeQuery::STATUS_INSERT (1) or MergeQuery::STATUS_UPDATE (2).
      if ($merge === \Drupal\Core\Database\Query\Merge::STATUS_INSERT) {
        if (\Drupal::config('wse_parallel.settings')->get('debug_mode')) {
          $this->logger->debug('Started editing session for @entity_type:@entity_id (@langcode)', [
            '@entity_type' => $entity_type,
            '@entity_id' => $entity_id,
            '@langcode' => $langcode,
          ]);
        }
      }
      else {
        if (\Drupal::config('wse_parallel.settings')->get('debug_mode')) {
          $this->logger->debug('Updated editing session for @entity_type:@entity_id (@langcode)', [
            '@entity_type' => $entity_type,
            '@entity_id' => $entity_id,
            '@langcode' => $langcode,
          ]);
        }
      }

      // Return a success indicator. We can't easily get the actual SID from merge,
      // but returning non-zero indicates success.
      return 1;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create/update session: @message', [
        '@message' => $e->getMessage(),
      ]);
      return 0;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getActiveSessions(ContentEntityInterface $entity): array {
    $entity_type = $entity->getEntityTypeId();
    $entity_id = $entity->id();
    $langcode = $entity->language()->getId();

    $sessions = $this->database->select('wse_parallel_edit_session', 's')
      ->fields('s')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->condition('langcode', $langcode)
      ->orderBy('last_seen', 'DESC')
      ->execute()
      ->fetchAllAssoc('sid', \PDO::FETCH_ASSOC);

    return $sessions ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function deleteSessionsForWorkspace(string $workspace_id): int {
    $num_deleted = $this->database->delete('wse_parallel_edit_session')
      ->condition('workspace_id', $workspace_id)
      ->execute();

    if ($num_deleted > 0 && \Drupal::config('wse_parallel.settings')->get('debug_mode')) {
      $this->logger->debug('Deleted @count sessions for workspace @workspace.', [
        '@count' => $num_deleted,
        '@workspace' => $workspace_id,
      ]);
    }

    return $num_deleted;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteSessionsForEntity(EntityInterface $entity): int {
    if (!$this->workspaceInformation->isEntityTypeSupported($entity->getEntityType())) {
      return 0;
    }

    $entity_type = $entity->getEntityTypeId();
    $entity_id = $entity->id();

    $num_deleted = $this->database->delete('wse_parallel_edit_session')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->execute();

    if ($num_deleted > 0 && \Drupal::config('wse_parallel.settings')->get('debug_mode')) {
      $this->logger->debug('Deleted @count sessions for entity @type:@id.', [
        '@count' => $num_deleted,
        '@type' => $entity_type,
        '@id' => $entity_id,
      ]);
    }

    return $num_deleted;
  }

  /**
   * Checks if an entity is a menu entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if the entity is a menu entity, FALSE otherwise.
   */
  protected function isMenuEntity(ContentEntityInterface $entity): bool {
    $menu_entity_types = ['menu', 'menu_link_content'];
    return in_array($entity->getEntityTypeId(), $menu_entity_types, TRUE);
  }

}
