<?php

namespace Drupal\wse_parallel\Parallel;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\workspaces\WorkspaceInterface;

/**
 * Interface for the session tracker service.
 */
interface SessionTrackerInterface {

  /**
   * Starts or updates an editing session for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity being edited.
   * @param int|null $baseRevisionId
   *   The revision ID the user started editing from.
   * @param int|null $editingRevisionId
   *   The current working revision ID.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace where editing is occurring.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account performing the edit.
   *
   * @return int
   *   The session ID (sid) of the created or updated session.
   */
  public function startOrUpdateSession(ContentEntityInterface $entity, ?int $baseRevisionId, ?int $editingRevisionId, WorkspaceInterface $workspace, AccountInterface $account): int;

  /**
   * Gets all active editing sessions for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check for active sessions.
   *
   * @return array
   *   An array of active session records.
   */
  public function getActiveSessions(ContentEntityInterface $entity): array;

  /**
   * Deletes all editing sessions for a workspace.
   *
   * @param string $workspace_id
   *   The workspace ID.
   *
   * @return int
   *   The number of sessions deleted.
   */
  public function deleteSessionsForWorkspace(string $workspace_id): int;

  /**
   * Deletes all editing sessions for a specific entity.
   *
   * @param EntityInterface $entity
   *   The entity being deleted.
   *
   * @return int
   *   The number of sessions deleted.
   */
  public function deleteSessionsForEntity(EntityInterface $entity): int;

}
