<?php

namespace Drupal\wse_parallel\Parallel;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\workspaces\WorkspaceInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for handling status inheritance when forking from initial revisions.
 */
class StatusInheritanceService {

  /**
   * Constructs a StatusInheritanceService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(protected Connection $database, protected EntityTypeManagerInterface $entityTypeManager, protected LoggerInterface $logger) {}

  /**
   * Checks and applies status inheritance if needed.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The current workspace.
   * @param int $base_revision_id
   *   The base revision ID from live.
   */
  public function checkAndApplyStatusInheritance(ContentEntityInterface $entity, WorkspaceInterface $workspace, int $base_revision_id): void {
    // Get the status field name from entity type keys.
    $status_field = $entity->getEntityType()->getKey('status');

    // Only applies to entities that have a status key defined.
    if (empty($status_field)) {
      return;
    }

    // Check if the entity has this field.
    if (!$entity->hasField($status_field)) {
      return;
    }

    // If already published, nothing to do.
    if ($entity->get($status_field)->value) {
      return;
    }

    // Check if this is the first time tracking this entity in this workspace.
    if (!$this->isFirstTracking($entity, $workspace)) {
      return;
    }

    // Check if the base (live) revision is the oldest/initial revision.
    if (!$this->isInitialRevision($entity, $base_revision_id)) {
      return;
    }

    // Check if the entity is tracked in any other workspace.
    if (!$this->isTrackedInOtherWorkspaces($entity, $workspace)) {
      return;
    }

    // All criteria met - set status to published.
    $entity->set($status_field, 1);

    $this->logger->info(
      'Inherited published status for @type:@id in workspace @workspace (was forked from unpublished initial revision)',
      [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
        '@workspace' => $workspace->label(),
      ]
    );
  }

  /**
   * Checks if this is the first time tracking an entity in a workspace.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The workspace.
   *
   * @return bool
   *   TRUE if this is the first tracking, FALSE otherwise.
   */
  protected function isFirstTracking(ContentEntityInterface $entity, WorkspaceInterface $workspace): bool {
    // Check if entity is already tracked in workspace_association for this workspace.
    $existing = $this->database->select('workspace_association', 'wa')
      ->fields('wa', ['target_entity_revision_id'])
      ->condition('workspace', $workspace->id())
      ->condition('target_entity_type_id', $entity->getEntityTypeId())
      ->condition('target_entity_id', $entity->id())
      ->range(0, 1)
      ->execute()
      ->fetchField();

    return empty($existing);
  }

  /**
   * Checks if a revision is the initial (oldest) revision for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   * @param int $revision_id
   *   The revision ID to check.
   *
   * @return bool
   *   TRUE if this is the initial revision, FALSE otherwise.
   */
  protected function isInitialRevision(ContentEntityInterface $entity, int $revision_id): bool {
    $storage = $this->entityTypeManager->getStorage($entity->getEntityTypeId());

    // If the entity type is not using core's default entity storage, we can't
    // assume the table mapping layout.
    if (!$storage instanceof SqlContentEntityStorage) {
      return FALSE;
    }

    $entity_type = $storage->getEntityType();
    $table_mapping = $storage->getTableMapping();
    $id_field = $table_mapping->getColumnNames($entity_type->getKey('id'))['value'];
    $revision_id_field = $table_mapping->getColumnNames($entity_type->getKey('revision'))['value'];

    // Get the lowest revision ID for this entity.
    $lowest_revision_id = $this->database->select($entity_type->getRevisionTable(), 'r')
      ->fields('r', [$revision_id_field])
      ->condition($id_field, $entity->id())
      ->orderBy($revision_id_field, 'ASC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    return $revision_id == $lowest_revision_id;
  }

  /**
   * Checks if an entity is tracked in any workspace other than the current one.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The current workspace.
   *
   * @return bool
   *   TRUE if tracked in other workspaces, FALSE otherwise.
   */
  protected function isTrackedInOtherWorkspaces(ContentEntityInterface $entity, WorkspaceInterface $workspace): bool {
    // Check workspace_association table for tracking in other workspaces.
    $other_workspace = $this->database->select('workspace_association', 'wa')
      ->fields('wa', ['workspace'])
      ->condition('target_entity_type_id', $entity->getEntityTypeId())
      ->condition('target_entity_id', $entity->id())
      ->condition('workspace', $workspace->id(), '!=')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    return !empty($other_workspace);
  }

}
