<?php

namespace Drupal\wse_parallel;

use Psr\Log\LoggerInterface;
use Drupal\navigation_plus\NavigationPlusUi;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Drupal\workspaces\WorkspaceInformationInterface;
use Drupal\wse_parallel\Parallel\ParallelWarningBuilder;
use Drupal\wse_parallel\Parallel\LiveRevisionResolverInterface;

/**
 * Service for displaying parallel edit warnings.
 */
class ParallelEditWarningService {

  /**
   * Constructs a ParallelEditWarningService object.
   *
   * @param \Drupal\navigation_plus\NavigationPlusUi $navigationPlusUi
   *   The Navigation+ UI service.
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager service.
   * @param \Drupal\workspaces\WorkspaceInformationInterface $workspaceInformation
   *   The workspace information service.
   * @param \Drupal\wse_parallel\Parallel\LiveRevisionResolverInterface $liveRevisionResolver
   *   The live revision resolver service.
   * @param \Drupal\wse_parallel\Parallel\ParallelWarningBuilder $warningBuilder
   *   The parallel warning builder service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected NavigationPlusUi $navigationPlusUi,
    protected WorkspaceManagerInterface $workspaceManager,
    protected WorkspaceInformationInterface $workspaceInformation,
    protected LiveRevisionResolverInterface $liveRevisionResolver,
    protected ParallelWarningBuilder $warningBuilder,
    protected MessengerInterface $messenger,
    protected LoggerInterface $logger,
  ) {}

  /**
   * Shows parallel edit warnings for the current page if applicable.
   */
  public function showWarningsForCurrentPage(): void {
    if ($this->navigationPlusUi->getMode() !== 'edit') {
      return;
    }

    $entity = $this->navigationPlusUi->deriveEntityFromRoute();

    if (!$entity || !($entity instanceof ContentEntityInterface)) {
      return;
    }

    if (!$this->workspaceInformation->isEntityTypeSupported($entity->getEntityType())) {
      return;
    }

    // Only show warnings if we have an active workspace.
    if (!$this->workspaceManager->hasActiveWorkspace()) {
      return;
    }

    try {
      $workspace = $this->workspaceManager->getActiveWorkspace();
      $base_revision_id = $this->liveRevisionResolver->getLiveRevisionId($entity);

      // Build and display warnings.
      $warnings = $this->warningBuilder->buildWarnings($entity, $workspace, $base_revision_id);

      if (!empty($warnings)) {
        foreach ($warnings as $warning) {
          $this->messenger->addWarning($warning);
        }
      }
    }
    catch (\Exception $e) {
      // Log but don't fail the page load.
      $this->logger->warning('Failed to show parallel edit warnings: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

}
