<?php

namespace Drupal\wse_parallel\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic local tasks for workspace history on content entities.
 */
class WorkspaceHistoryLocalTask extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * Constructs a WorkspaceHistoryLocalTask object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager.
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager, protected WorkspaceManagerInterface $workspaceManager) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('workspaces.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $this->derivatives = [];

    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      // Only add tabs for entity types that have a canonical route.
      if (!$entity_type->hasLinkTemplate('canonical')) {
        continue;
      }

      // Only add tabs for entity types supported by workspaces.
      if (!$this->workspaceManager->isEntityTypeSupported($entity_type)) {
        continue;
      }

      $this->derivatives["$entity_type_id.workspace_history"] = [
        'route_name' => "entity.$entity_type_id.workspace_history",
        'title' => $this->t('Workspace History'),
        'base_route' => "entity.$entity_type_id.canonical",
        'weight' => 100,
      ] + $base_plugin_definition;
    }

    return $this->derivatives;
  }

}
