<?php

namespace Drupal\wse_parallel\Plugin\WseParallelConflictStrategy;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\wse_parallel\Conflict\ConflictResolutionResult;
use Drupal\wse_parallel\Conflict\ConflictStrategyInterface;

/**
 * Provides a 'Last Publish Wins' conflict resolution strategy.
 *
 * This strategy does not perform any merging. The most recently published
 * workspace changes simply become the default/live revision. Earlier
 * unpublished changes remain as workspace-only revisions.
 *
 * @WseParallelConflictStrategy(
 *   id = "last_publish_wins",
 *   label = @Translation("Last Publish Wins"),
 *   description = @Translation("The most recently published workspace changes take precedence. No automatic merging is performed."),
 *   weight = 0
 * )
 */
class LastPublishWins extends PluginBase implements ConflictStrategyInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function onPrePublish(array $context): ConflictResolutionResult {
    // For last publish wins, we don't need to do anything special before
    // publish. Core workspaces will handle making the workspace revision
    // the new default.
    return ConflictResolutionResult::success(
      $this->t('Pre-publish validation passed for Last Publish Wins strategy.'),
      'validated'
    );
  }

  /**
   * {@inheritdoc}
   */
  public function onPostPublish(array $context): ConflictResolutionResult {
    // After publish, the workspace revisions are now the default revisions.
    // Any other workspace revisions that were based on the old default
    // are now diverged, but we don't modify them.
    
    $entity_count = 0;
    if (isset($context['published_revision_ids'])) {
      foreach ($context['published_revision_ids'] as $entity_type_revisions) {
        $entity_count += count($entity_type_revisions);
      }
    }

    return ConflictResolutionResult::success(
      $this->t('Published @count entities using Last Publish Wins strategy.', [
        '@count' => $entity_count,
      ]),
      'published',
      ['entity_count' => $entity_count]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function resolve(ContentEntityInterface $entity, array $context): ConflictResolutionResult {
    // For last publish wins, resolution is simple: the workspace revision
    // becomes the default. No merging or conflict detection needed.
    
    return ConflictResolutionResult::success(
      $this->t('Resolved using Last Publish Wins: workspace revision becomes default.'),
      'overwrite',
      [
        'entity_id' => $entity->id(),
        'entity_type' => $entity->getEntityTypeId(),
        'workspace_revision_id' => $context['workspace_revision_id'] ?? NULL,
      ]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    return (string) $this->t('Last Publish Wins');
  }

  /**
   * {@inheritdoc}
   */
  public function description(): string {
    return (string) $this->t('The most recently published workspace changes take precedence. No automatic merging is performed. Earlier unpublished changes remain as workspace-only revisions.');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Last Publish Wins has no configuration options.
    $form['info'] = [
      '#markup' => '<p>' . $this->description() . '</p>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    // No validation needed.
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // No configuration to save.
  }

}
