<?php

namespace Drupal\wse_parallel\Publish;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableStorageInterface;
use Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\workspaces\WorkspaceInterface;
use Drupal\workspaces\WorkspaceManagerInterface;

/**
 * Service for logging publish events with from/to revision pairs.
 */
class PublishLogger implements PublishLoggerInterface {

  /**
   * Constructs a PublishLogger object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface $keyValueExpirable
   *   The key-value expirable factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager.
   */
  public function __construct(protected Connection $database, protected EntityTypeManagerInterface $entityTypeManager, protected KeyValueExpirableFactoryInterface $keyValueExpirable, protected TimeInterface $time, protected WorkspaceManagerInterface $workspaceManager) {}

  /**
   * {@inheritdoc}
   */
  public function recordPrePublish(WorkspaceInterface $workspace, array $publishedRevisionIds): void {
    $from_by_entity = [];

    // Temporarily switch to the live workspace to get the current live revisions.
    $original_workspace = $this->workspaceManager->getActiveWorkspace();

    try {
      // Switch to live workspace to get "from" revisions.
      $this->workspaceManager->switchToLive();

      foreach ($publishedRevisionIds as $entity_type_id => $revision_map) {
        // Skip non-revisionable entity types.
        $storage = $this->entityTypeManager->getStorage($entity_type_id);
        if (!$storage instanceof RevisionableStorageInterface) {
          continue;
        }

        // Extract entity IDs from the revision map.
        $entity_ids = array_values($revision_map);
        if (empty($entity_ids)) {
          continue;
        }

        // Load the current default revisions (from live workspace).
        try {
          $default_entities = $storage->loadMultiple($entity_ids);

          foreach ($default_entities as $entity_id => $entity) {
            if ($entity && method_exists($entity, 'getRevisionId')) {
              $from_by_entity[$entity_type_id][$entity_id] = $entity->getRevisionId();
            }
          }
        }
        catch (EntityStorageException $e) {
          // Log error but continue processing other entity types.
          \Drupal::logger('wse_parallel')->error(
            'Failed to load default revisions for @type during pre-publish: @message',
            [
              '@type' => $entity_type_id,
              '@message' => $e->getMessage(),
            ]
          );
        }
      }
    }
    finally {
      // Always restore the original workspace, even if an error occurred.
      if ($original_workspace) {
        $this->workspaceManager->setActiveWorkspace($original_workspace);
      }
    }

    // Store the pre-publish state in key-value expirable storage.
    $kv = $this->keyValueExpirable->get('wse_parallel.pre_publish');
    $key = 'workspace:' . $workspace->id();

    $kv->setWithExpire($key, [
      'created' => $this->time->getRequestTime(),
      'from_by_entity' => $from_by_entity,
    ], 900); // 15 minutes TTL
  }

  /**
   * {@inheritdoc}
   */
  public function recordPostPublish(WorkspaceInterface $workspace, array $publishedRevisionIds): void {
    $workspace_id = $workspace->id();
    $published_time = $this->time->getRequestTime();

    // Start a database transaction to ensure atomic logging.
    $transaction = $this->database->startTransaction();

    try {
      // Try to retrieve the pre-publish state.
      $kv = $this->keyValueExpirable->get('wse_parallel.pre_publish');
      $key = 'workspace:' . $workspace_id;
      $pre_state = $kv->get($key);
      
      $from_by_entity = $pre_state['from_by_entity'] ?? [];

      // Process each entity type.
      foreach ($publishedRevisionIds as $entity_type_id => $revision_map) {
        // Skip non-revisionable entity types.
        $storage = $this->entityTypeManager->getStorage($entity_type_id);
        if (!$storage instanceof RevisionableStorageInterface) {
          continue;
        }

        if (empty($revision_map)) {
          continue;
        }

        // Collect all revision IDs to batch load.
        $revision_ids = array_keys($revision_map);

        // Batch load all revisions at once.
        $loaded_revisions = [];
        try {
          $loaded_revisions = $storage->loadMultipleRevisions($revision_ids);
        }
        catch (EntityStorageException $e) {
          // Log error but continue - langcode will be NULL for failed loads.
          \Drupal::logger('wse_parallel')->warning(
            'Failed to batch load revisions for @type: @message',
            ['@type' => $entity_type_id, '@message' => $e->getMessage()]
          );
        }

        // Prepare batch insert.
        $insert_values = [];

        foreach ($revision_map as $to_revision_id => $entity_id) {
          // Get from_revision_id from pre-state, or NULL if not available.
          $from_revision_id = $from_by_entity[$entity_type_id][$entity_id] ?? NULL;

          // Get langcode from pre-loaded revision if available.
          $langcode = NULL;
          if (isset($loaded_revisions[$to_revision_id])) {
            $entity = $loaded_revisions[$to_revision_id];
            if ($entity && method_exists($entity, 'language')) {
              $langcode = $entity->language()->getId();
            }
          }

          $insert_values[] = [
            'entity_type' => $entity_type_id,
            'entity_id' => $entity_id,
            'langcode' => $langcode,
            'from_revision_id' => $from_revision_id ?? 0,
            'to_revision_id' => $to_revision_id,
            'workspace_id' => $workspace_id,
            'published' => $published_time,
          ];
        }

        // Batch insert all records for this entity type.
        if (!empty($insert_values)) {
          $query = $this->database->insert('wse_parallel_publish_log')
            ->fields([
              'entity_type',
              'entity_id',
              'langcode',
              'from_revision_id',
              'to_revision_id',
              'workspace_id',
              'published',
            ]);

          foreach ($insert_values as $values) {
            $query->values($values);
          }

          $query->execute();
        }
      }

      // Clean up the pre-publish state.
      $kv->delete($key);

      // Transaction commits implicitly when $transaction goes out of scope.
    }
    catch (\Exception $e) {
      // Catch all exceptions to ensure transaction rollback.
      // This includes DatabaseException, EntityStorageException, and others.
      $transaction->rollBack();

      \Drupal::logger('wse_parallel')->error(
        'Failed to log publish event for workspace @workspace: @message',
        [
          '@workspace' => $workspace_id,
          '@message' => $e->getMessage(),
        ]
      );

      // Re-throw the exception so the publish operation knows logging failed.
      throw $e;
    }
  }

}
