<?php

namespace Drupal\wse_parallel\Publish;

use Drupal\Core\Database\Connection;

/**
 * Service for looking up publish log records.
 */
class PublishLookup {

  /**
   * Constructs a PublishLookup object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(protected Connection $database) {}

  /**
   * Returns TRUE if any publish exists for (type, id) after $timestamp.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity_id
   *   The entity ID.
   * @param int $timestamp
   *   The timestamp to check after.
   *
   * @return bool
   *   TRUE if a publish exists after the timestamp, FALSE otherwise.
   */
  public function hasPublishSince(string $entity_type, int|string $entity_id, int $timestamp): bool {
    $count = $this->database->select('wse_parallel_publish_log', 'p')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->condition('published', $timestamp, '>')
      ->countQuery()
      ->execute()
      ->fetchField();

    return $count > 0;
  }

  /**
   * Returns the latest publish record for an entity.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity_id
   *   The entity ID.
   *
   * @return array|null
   *   The publish record array with keys: to_revision_id, from_revision_id,
   *   workspace_id, published, langcode. NULL if no record found.
   */
  public function latestPublish(string $entity_type, int|string $entity_id): ?array {
    $record = $this->database->select('wse_parallel_publish_log', 'p')
      ->fields('p', [
        'to_revision_id',
        'from_revision_id',
        'workspace_id',
        'published',
        'langcode',
        'entity_type',
        'entity_id',
      ])
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->orderBy('published', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchAssoc();

    return $record ?: NULL;
  }

  /**
   * Returns all publish records for an entity.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity_id
   *   The entity ID.
   * @param int|null $limit
   *   Optional limit on number of records to return.
   * @param int $offset
   *   Optional offset for pagination.
   *
   * @return array
   *   Array of publish records, ordered by published timestamp descending.
   */
  public function getPublishHistory(string $entity_type, int|string $entity_id, ?int $limit = NULL, int $offset = 0): array {
    $query = $this->database->select('wse_parallel_publish_log', 'p')
      ->fields('p')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->orderBy('published', 'DESC');

    if ($limit !== NULL) {
      $query->range($offset, $limit);
    }

    return $query->execute()->fetchAllAssoc('plid', \PDO::FETCH_ASSOC);
  }

  /**
   * Returns the total count of publish records for an entity.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity_id
   *   The entity ID.
   *
   * @return int
   *   The count of publish records.
   */
  public function getPublishHistoryCount(string $entity_type, int|string $entity_id): int {
    return (int) $this->database->select('wse_parallel_publish_log', 'p')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

}
