<?php

namespace Drupal\wse_parallel\Routing;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\workspaces\WorkspaceManagerInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;
use Drupal\Core\Routing\RouteSubscriberBase;

/**
 * Provides dynamic routes for workspace history on content entities.
 */
class WorkspaceHistoryRoutes extends RouteSubscriberBase {

  /**
   * Constructs a WorkspaceHistoryRoutes object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspaceManager
   *   The workspace manager.
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager, protected WorkspaceManagerInterface $workspaceManager) {}

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection) {
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      // Only add routes for entity types that have a canonical route.
      if (!$entity_type->hasLinkTemplate('canonical')) {
        continue;
      }

      // Only add routes for entity types supported by workspaces.
      if (!$this->workspaceManager->isEntityTypeSupported($entity_type)) {
        continue;
      }

      // Get the canonical route to find its path.
      $canonical_route_name = "entity.$entity_type_id.canonical";
      $canonical_route = $collection->get($canonical_route_name);

      if (!$canonical_route) {
        continue;
      }

      // Get the canonical route's path and append our suffix.
      $path = $canonical_route->getPath() . '/workspace-history';

      // Create the workspace history route for this entity type.
      $route = new Route(
        $path,
        [
          '_controller' => '\Drupal\wse_parallel\Controller\WorkspaceHistoryController::history',
          '_title_callback' => '\Drupal\wse_parallel\Controller\WorkspaceHistoryController::title',
          'entity_type' => $entity_type_id,
        ],
        [
          '_permission' => 'view workspace history',
        ],
        [
          'parameters' => [
            $entity_type_id => [
              'type' => 'entity:' . $entity_type_id,
            ],
          ],
          '_admin_route' => TRUE,
        ]
      );

      $collection->add("entity.$entity_type_id.workspace_history", $route);
    }
  }

}
