<?php

namespace Drupal\wse_parallel\Snapshot;

use Drupal\Core\Database\Connection;
use Drupal\workspaces\WorkspaceInterface;

/**
 * Service to manage workspace snapshots for time machine functionality.
 */
class SnapshotManager implements SnapshotManagerInterface {

  /**
   * Constructs a SnapshotManager object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(protected Connection $database) {}

  /**
   * {@inheritdoc}
   */
  public function createSnapshot(WorkspaceInterface $workspace, array $published_revision_ids): void {
    // Get the most recent closed workspace before this one to use as base.
    $previous_snapshot_data = $this->getPreviousSnapshot($workspace);

    // Start with previous snapshot or empty array.
    $snapshot_data = $previous_snapshot_data ?? [];

    // Merge in the newly published revisions.
    // $published_revision_ids format: [entity_type => [revision_id => entity_id]]
    foreach ($published_revision_ids as $entity_type => $revisions) {
      if (!isset($snapshot_data[$entity_type])) {
        $snapshot_data[$entity_type] = [];
      }

      foreach ($revisions as $revision_id => $entity_id) {
        $snapshot_data[$entity_type][$entity_id] = $revision_id;
      }
    }

    // Save the snapshot.
    $this->database->insert('wse_parallel_workspace_snapshot')
      ->fields([
        'workspace_id' => $workspace->id(),
        'snapshot_data' => json_encode($snapshot_data),
        'published_timestamp' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function deleteSnapshot(string $workspace_id): void {
    $this->database->delete('wse_parallel_workspace_snapshot')
      ->condition('workspace_id', $workspace_id)
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getSnapshot(string $workspace_id): ?array {
    $snapshot_json = $this->database->select('wse_parallel_workspace_snapshot', 's')
      ->fields('s', ['snapshot_data'])
      ->condition('workspace_id', $workspace_id)
      ->execute()
      ->fetchField();

    if ($snapshot_json) {
      return json_decode($snapshot_json, TRUE);
    }

    return NULL;
  }

  /**
   * Gets the snapshot from the most recently published workspace before this one.
   *
   * @param \Drupal\workspaces\WorkspaceInterface $workspace
   *   The current workspace.
   *
   * @return array|null
   *   The previous snapshot data, or NULL if none exists.
   */
  protected function getPreviousSnapshot(WorkspaceInterface $workspace): ?array {
    // Find the most recent closed workspace before this one.
    $previous_workspace_id = $this->database->select('workspace', 'w')
      ->fields('w', ['id'])
      ->condition('status', 'closed')
      ->condition('changed', $workspace->getChangedTime(), '<')
      ->orderBy('changed', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    if ($previous_workspace_id) {
      return $this->getSnapshot($previous_workspace_id);
    }

    return NULL;
  }

}
