<?php

namespace Drupal\wse_parallel\Validation;

use Drupal\Core\Validation\ConstraintManager;
use Drupal\workspaces\WorkspaceManagerInterface;

/**
 * Decorates the constraint manager to filter out workspace conflict constraints.
 */
class ParallelConstraintManager extends ConstraintManager {

  /**
   * The decorated constraint manager.
   *
   * @var \Drupal\Core\Validation\ConstraintManager
   */
  protected $innerManager;

  /**
   * The workspace manager.
   *
   * @var \Drupal\workspaces\WorkspaceManagerInterface
   */
  protected $workspaceManager;

  /**
   * Constructs a ParallelConstraintManager.
   *
   * @param \Drupal\Core\Validation\ConstraintManager $inner_manager
   *   The decorated constraint manager.
   * @param \Drupal\workspaces\WorkspaceManagerInterface $workspace_manager
   *   The workspace manager.
   */
  public function __construct(ConstraintManager $inner_manager, WorkspaceManagerInterface $workspace_manager) {
    $this->innerManager = $inner_manager;
    $this->workspaceManager = $workspace_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function create($plugin_id, $configuration = []) {
    // If we're in a workspace and the constraint is EntityWorkspaceConflict,
    // return a no-op constraint instead.
    if ($this->workspaceManager->getActiveWorkspace() && $plugin_id === 'EntityWorkspaceConflict') {
      // Return a constraint that always passes.
      return new \Drupal\wse_parallel\Validation\NoOpConstraint();
    }

    return $this->innerManager->create($plugin_id, $configuration);
  }

  /**
   * Pass through all other method calls to the inner manager.
   */
  public function __call($method, $args) {
    return call_user_func_array([$this->innerManager, $method], $args);
  }

}
